﻿using System;
using System.IO;
using System.Text;
using System.Collections.Generic;
using System.Windows.Forms;
using System.Security.Cryptography;
using System.Drawing;
using System.Drawing.Imaging;
using System.Runtime.InteropServices;

namespace YumeNikkiRandomizer
{
    class M
    {
        public static int versionID = 105; // Must match base game version in folder name and Game Start
        public static string versionStr = "1.05"; // For display and hash only; letters indicate minor updates to program but same base game
        
        #region // Data Constants //
        
        // Constants for each resource folder and its mode.
        public const int M_BACKDROP = 0;
        public const int M_BATTLE = 1;
        public const int M_BATTLE2 = 2;
        public const int M_BATTLECHARSET = 3;
        public const int M_BATTLEWEAPON = 4;
        public const int M_CHARSET = 5;
        public const int M_CHIPSET = 6;
        public const int M_FACESET = 7;
        public const int M_FRAME = 8;
        public const int M_GAMEOVER = 9;
        public const int M_MONSTER = 10;
        public const int M_MOVIE = 11;
        public const int M_MUSIC = 12;
        public const int M_PANORAMA = 13;
        public const int M_PICTURE = 14;
        public const int M_SOUND = 15;
        public const int M_SYSTEM = 16;
        public const int M_SYSTEM2 = 17;
        public const int M_TITLE = 18;
        public const int FOLDERCOUNT = 19;
        
        // Constants for other modes.
        public const int M_MESSAGEALL = 19;
        public const int M_MESSAGEPART = 20;
        public const int M_MESSAGESTART = 21;
        public const int M_OPTION = 22;
        public const int M_NAME = 23;
        public const int M_NICKNAME = 24;
        public const int M_COMMENT = 25;
        public const int MODECOUNT = 26;
        
        // Constant for all folder/mode names.
        public static string[] FOLDER = {
                                            "Backdrop",
                                            "Battle",
                                            "Battle2",
                                            "BattleCharSet",
                                            "BattleWeapon",
                                            "CharSet",
                                            "ChipSet",
                                            "FaceSet",
                                            "Frame",
                                            "GameOver",
                                            "Monster",
                                            "Movie",
                                            "Music",
                                            "Panorama",
                                            "Picture",
                                            "Sound",
                                            "System",
                                            "System2",
                                            "Title",
                                            "MessageAll",
                                            "MessagePart",
                                            "MessageStart",
                                            "Option",
                                            "Name",
                                            "Nickname",
                                            "Comment"
                                        };
        
        // Constants for string categories.
        public static int S_TOTRANSLATE = 0;
        public static int S_UNTRANSLATED = 1;
        public static int S_FILENAME = 2;
        public static int S_CONSTANT = 3;
        #endregion
        
        #region // Yume Nikki Constants //
        
        // Effect item IDs.
        public const int EF_FROG = 1;
        public const int EF_UMBRELLA = 3;
        public const int EF_HATANDSCARF = 5;
        public const int EF_YUKIONNA = 7;
        public const int EF_KNIFE = 9;
        public const int EF_EYEBALLHAND = 11;
        public const int EF_FAT = 13;
        public const int EF_SMALL = 15;
        public const int EF_FLUTE = 17;
        public const int EF_NEON = 19;
        public const int EF_NOPPERABOU = 21;
        public const int EF_SEVEREDHEAD = 23;
        public const int EF_TOWEL = 25;
        public const int EF_CAT = 27;
        public const int EF_LAMP = 29;
        public const int EF_BICYCLE = 31;
        public const int EF_LONGHAIR = 33;
        public const int EF_POOPHAIR = 35;
        public const int EF_BLONDEHAIR = 37;
        public const int EF_TRIANGLEKERCHIEF = 39;
        public const int EF_WITCH = 41;
        public const int EF_ONI = 43;
        public const int EF_SQUISHY = 45;
        public const int EF_STOPLIGHT = 47;
        
        public static List<int> allEffects = new List<int> { EF_FROG, EF_UMBRELLA, EF_HATANDSCARF, EF_YUKIONNA,
                                                             EF_KNIFE, EF_EYEBALLHAND, EF_FAT, EF_SMALL,
                                                             EF_FLUTE, EF_NEON, EF_NOPPERABOU, EF_SEVEREDHEAD,
                                                             EF_TOWEL, EF_CAT, EF_LAMP, EF_BICYCLE,
                                                             EF_LONGHAIR, EF_POOPHAIR, EF_BLONDEHAIR, EF_TRIANGLEKERCHIEF,
                                                             EF_WITCH, EF_ONI, EF_SQUISHY, EF_STOPLIGHT };
        
        // Essence item IDs.
        public const int ES_UBOA = 49;
        public const int ES_MONOE = 50;
        public const int ES_MONOKO = 51;
        public const int ES_MASADA = 52;
        public const int ES_FACE = 53;
        public const int ES_FLIGHT = 54;
        public const int ES_PARTY = 55;
        public const int ES_NASU = 56;
        
        public static List<int> allEssences = new List<int> { ES_UBOA, ES_MONOE, ES_MONOKO, ES_MASADA,
                                                              ES_FACE, ES_FLIGHT, ES_PARTY, ES_NASU };
        
        // Map IDs.
        public const int MAP_TESTMAP = 1;
        public const int MAP_GAMESTART = 2;
        public const int MAP_REALWORLDROOM = 3;
        public const int MAP_VERANDA = 4;
        // 5: N/A
        public const int MAP_DREAMROOM = 6;
        public const int MAP_DREAMVERANDAA = 7;
        public const int MAP_DREAMVERANDAB = 8;
        public const int MAP_NEXUS = 9;
        public const int MAP_FORESTWORLD = 10;
        public const int MAP_BLOCKWORLD = 11;
        public const int MAP_PUDDLEWORLD = 12;
        public const int MAP_DARKWORLD = 13;
        public const int MAP_SNOWWORLD = 14;
        public const int MAP_MURALWORLD = 15;
        public const int MAP_GRAFFITIWORLD = 16;
        public const int MAP_EYEBALLWORLD = 17;
        public const int MAP_CANDLEWORLD = 18;
        public const int MAP_SHIELDFOLKWORLD = 19;
        public const int MAP_NEONWORLD = 20;
        public const int MAP_NUMBERWORLD = 21;
        public const int MAP_IGLOO1 = 22;
        public const int MAP_IGLOO2 = 23;
        public const int MAP_IGLOO3 = 24;
        public const int MAP_IGLOO4 = 25;
        public const int MAP_IGLOO5 = 26;
        public const int MAP_IGLOO6 = 27;
        public const int MAP_IGLOO7 = 28;
        public const int MAP_STABBINGROOM = 29;
        public const int MAP_ROOMOFBEDS = 30;
        public const int MAP_LAMPWORLD = 31;
        public const int MAP_FACESTAIRWAYPASSAGE = 32;
        public const int MAP_FACESTAIRWAY = 33;
        public const int MAP_FACE = 34;
        public const int MAP_GUILLOTINEWORLDSMALL = 35;
        public const int MAP_GUILLOTINEWORLDBIG = 36;
        public const int MAP_NEONTILESA = 37;
        public const int MAP_NEONTILESB = 38;
        public const int MAP_CHECKERTILESA = 39;
        public const int MAP_CHECKERTILESB = 40;
        public const int MAP_FACECARPETPLAZA = 41;
        public const int MAP_HIGHWAY = 42;
        public const int MAP_HIGHWAYA = 43;
        public const int MAP_HIGHWAYB = 44;
        public const int MAP_HIGHWAYC = 45;
        public const int MAP_HIGHWAYD = 46;
        public const int MAP_HELL = 47;
        public const int MAP_FOOTPRINTPATHA = 48;
        public const int MAP_FOOTPRINTPATHB = 49;
        public const int MAP_WILDERNESSGATE = 50;
        public const int MAP_WILDERNESSROUTE4 = 51;
        public const int MAP_WILDERNESSROUTE1 = 52;
        public const int MAP_WILDERNESSROUTE2 = 53;
        public const int MAP_WILDERNESSRAVEBOX = 54;
        public const int MAP_WILDERNESSPARTY = 55;
        public const int MAP_WILDERNESSTOSETTLEMENT = 56;
        public const int MAP_WILDERNESSFCPORTAL = 57;
        public const int MAP_WILDERNESSROUTE3 = 58;
        public const int MAP_WILDERNESSGAPINFENCE = 59;
        public const int MAP_INFINITEWILDERNESS = 60;
        public const int MAP_WILDERNESSFENCE = 61;
        public const int MAP_WILDERNESSPURPLEPYLONS = 62;
        public const int MAP_WILDERNESSFOOTOFSTAIRS = 63;
        public const int MAP_STAIRWAYTOTHESKY = 64;
        public const int MAP_INSIDESHACKA = 65;
        public const int MAP_BARRACKSSETTLEMENT = 66;
        public const int MAP_MINIHELL = 67;
        public const int MAP_BARRACKSSHACKA = 68;
        public const int MAP_BARRACKSSHACKB = 69;
        public const int MAP_GUTTER001 = 70;
        public const int MAP_GUTTER002 = 71;
        public const int MAP_GUTTER003 = 72;
        public const int MAP_GUTTER004 = 73;
        public const int MAP_GUTTER005 = 74;
        public const int MAP_GUTTER006 = 75;
        public const int MAP_MALLA = 76;
        public const int MAP_MALLB = 77;
        public const int MAP_MALLRECEPTION = 78;
        public const int MAP_MALLMUSIC = 79;
        public const int MAP_MALLTOKUTO = 80;
        public const int MAP_BIGRED = 81;
        public const int MAP_MALLC = 82;
        public const int MAP_WINDMILLWORLD = 83;
        public const int MAP_JUKAI1 = 84;
        // 85: Sub-Maps
        public const int MAP_JUKAI2 = 86;
        public const int MAP_JUKAITRAINBEFORE = 87;
        public const int MAP_TRAININTERIOR = 88;
        public const int MAP_JUKAITRAINAFTER = 89;
        public const int MAP_JUKAI3 = 90;
        public const int MAP_JUKAI4 = 91;
        public const int MAP_WITCHBRIDGE = 92;
        public const int MAP_WITCHISLAND = 93;
        public const int MAP_MONO = 94;
        public const int MAP_MONO001 = 95;
        public const int MAP_MONO002 = 96;
        public const int MAP_MONO003 = 97;
        public const int MAP_MONO004 = 98;
        public const int MAP_MONO005 = 99;
        public const int MAP_MONO006 = 100;
        public const int MAP_MONO007 = 101;
        public const int MAP_MONO008 = 102;
        public const int MAP_MONO009 = 103;
        public const int MAP_MONO010 = 104;
        public const int MAP_MONO011 = 105;
        public const int MAP_MONO012 = 106;
        public const int MAP_MONO013 = 107;
        public const int MAP_PINKSEA = 108;
        public const int MAP_PONIKOHOUSE = 109;
        public const int MAP_WARPMAZE = 110;
        public const int MAP_DOCKS = 111;
        public const int MAP_PARK = 112;
        public const int MAP_PARKCLIFF = 113;
        public const int MAP_GHOSTWORLD = 114;
        public const int MAP_INSIDEBRICKSA = 115;
        public const int MAP_INSIDEBRICKSB = 116;
        public const int MAP_INSIDEBRICKSC = 117;
        public const int MAP_FCENTRANCEA = 118;
        // 119: FC Maps
        public const int MAP_FCHOUSEOUTSIDE = 120;
        public const int MAP_FCHOUSEINSIDE = 121;
        public const int MAP_FCBASEMENT001 = 122;
        public const int MAP_FCBASEMENT002 = 123;
        public const int MAP_FCBASEMENT003 = 124;
        public const int MAP_FCBASEMENT004 = 125;
        public const int MAP_FCBASEMENT005 = 126;
        public const int MAP_FCBASEMENT006 = 127;
        public const int MAP_FCBASEMENT007 = 128;
        public const int MAP_FCBASEMENT008 = 129;
        public const int MAP_FCBASEMENT009 = 130;
        public const int MAP_FCBASEMENT010 = 131;
        public const int MAP_FCBASEMENT011 = 132;
        public const int MAP_FCBASEMENT012 = 133;
        public const int MAP_FCBASEMENT013 = 134;
        public const int MAP_FCBASEMENT014 = 135;
        public const int MAP_FCBASEMENT015 = 136;
        public const int MAP_FCBASEMENT016 = 137;
        public const int MAP_FCBASEMENT017 = 138;
        public const int MAP_FCBASEMENT018 = 139;
        public const int MAP_FCBASEMENT019 = 140;
        public const int MAP_FCBASEMENT020 = 141;
        public const int MAP_FCBASEMENT021 = 142;
        public const int MAP_FCBASEMENT022 = 143;
        public const int MAP_FCENTRANCEB = 144;
        public const int MAP_FCCAVEMOUTH = 145;
        public const int MAP_FCPYRAMIDS = 146;
        public const int MAP_FCDUNGEONVILLAGE = 147;
        public const int MAP_FCDUNGEON001 = 148;
        public const int MAP_FCDUNGEON002 = 149;
        public const int MAP_FCDUNGEON003 = 150;
        public const int MAP_FCLIZARDVILLAGE = 151;
        public const int MAP_FCUNDERVILLAGE = 152;
        public const int MAP_FCSMALLMAP003 = 153; // Unused
        public const int MAP_FCPIRORIVILLAGE = 154;
        public const int MAP_NASU = 155;
        public const int MAP_ENDING = 156;
        public const int MAP_FCDUNGEON002GLITCH = 157;
        public const int MAP_FCOVERWORLD = 158;
        public const int MAP_STAIRCASEOFHANDS = 159;
        public const int MAP_BEFOREBLAZINGCORRIDOR = 160;
        public const int MAP_BLAZINGCORRIDOR = 161;
        public const int MAP_STORAGEROOM = 162;
        public const int MAP_SPACESHIPENTRANCE = 163;
        public const int MAP_SPACESHIP01 = 164;
        public const int MAP_SPACESHIP02 = 165;
        public const int MAP_SPACESHIP03 = 166;
        public const int MAP_SPACESHIPEXIT = 167;
        public const int MAP_MARS01 = 168;
        public const int MAP_MARS02 = 169;
        public const int MAP_MARS03 = 170;
        public const int MAP_MARS04 = 171;
        public const int MAP_MARS05 = 172;
        public const int MAP_MARSBASEMENTSTAIRS = 173;
        public const int MAP_MARSBASEMENT = 174;
        // 175: N/A
        // 176: N/A
        public const int MAP_MALLD = 177;
        public const int MAP_MALLROOF = 178;
        public const int MAP_WITCHFLIGHT = 179;
        
        // Directions.
        public const int D_SAME = -1;
        public const int D_UP = 0;
        public const int D_RIGHT = 1;
        public const int D_DOWN = 2;
        public const int D_LEFT = 3;
        #endregion
        
        #region // Variables //
        
        // References for encodings.
        public static Encoding UNICODE;
        public static Encoding[] readEncodings;
        public static Encoding[] writeEncodings;
        public static int[] readEncodingIDs;
        public static int[] writeEncodingIDs;
        
        // Indicators of "where" the program currently is.
        public static string gamePath = "";
        public static string currentFile = "";
        public static string currentEvent = "";
        public static string currentPage = "";
        public static string currentLine = "";
        public static int currentEventNum = 0;
        public static int currentPageNum = 0;
        public static int currentBattlerAnimation = 0;
        
        // Game data.
        public static MapTree mapTree;
        public static Database database;
        public static Dictionary<int, Map> maps;
        public static int[][] heckTiles;
        
        // User options.
        public static string currentSeed;
        public static MD5 md5Hasher;
        public static Int32 trueSeed;
        public static Random random;
        
        public static bool randomizeEffects = true;
        public static int nexusLockCount = 8;
        public static int entranceRandoType = 1;
        public static int bicyclePlacement = 0;
        public static int eyeballPlacement = 0;
        public static bool nexusLockHints = false;
        public static bool spoilerLogEnabled = false;
        
        public static int effectsGoal = 24;
        public static int essencesGoal = 0;
        public static int minimumEffects = 24;
        public static int minimumEssences = 0;
        
        public static bool derandomizeEvents = true;
        public static bool fixChairGlitch = true;
        public static int walkSpeed = 4;
        public static int bikeSpeed = 5;
        public static int walkSound = 0;
        public static int heckificationType = 1;
        public static bool easyTeleporterMaze = true;
        public static int darknessLevel = 0;
        public static bool giversReflectEffect = false;
        
        // Display names for multi-choice user options.
        public static string[] entranceRandoTypes = { "None", "Nexus Doors", "Nexus Doors + Return", "Nightmare", "Complete Nightmare" };
        public static string[] itemPlacements = { "Normal", "Early", "Start With" };
        public static string[] walkSounds = { "Normal", "Carpet", "None" };
        public static string[] heckificationTypes = { "Hell", "Heck", "And Goodbye" };
        public static string[] darknessLevels = { "Normal", "Less Dark", "Pitch Black" };
        
        // Variables for game layout.
        public static Dictionary<string, CommandLocation> warpLocations;
        public static Dictionary<int, List<string>> warpKeysForMap;
        public static Dictionary<string, Warp> exits;
        public static List<WarpLink> linkedExits;
        
        public static List<string> noShuffleKeys;
        public static string[] nexusExitKeys;
        public static string[] backToNexusKeys;
        public static string[] intoHellKeys;
        public static string[] outtaHellKeys;
        
        public static Dictionary<string, CommandLocation> effectLocations;
        public static Dictionary<int, List<string>> effectKeysForMap;
        public static Dictionary<string, string> effectLocationText;
        public static Dictionary<string, int> effects;
        
        public static Dictionary<string, CommandLocation> essenceLocations;
        public static Dictionary<int, List<string>> essenceKeysForMap;
        public static Dictionary<string, int> essences;
        
        public static List<int> mapsWithDreamBeds;
        public static List<int> mapsInFCWorld;
        public static List<int> mapsWithSnow;
        public static List<int> mapsWithRain;
        public static List<int> mapsWithDarkness;
        
        public static int[] nexusLocks;
        
        public static int fixedDreamBedID;
        public static int fixedClosetWarpID;
        public static int fixedMiniHellClosetID;
        public static int fixedPurpleVersion;
        public static int fixedPurpleClosetSwap;
        public static int fixedVillageWarper;
        public static bool fixedGlitchEvent;
        
        // Variables for simulated playthrough to test validity.
        public static List<int> simulationEffects;
        public static List<int> pendingEffects;
        public static List<int> simulationEssences;
        public static List<int> pendingEssences;
        public static List<string> simulationMapsChecked;
        public static bool simulationProgressMade = false;
        
        // Variables for finding Effects and Essences at their lowest search depths.
        public static bool findingItemLocationsByDepth = false;
        public static Dictionary<int, List<string>> effectLocationsAtDepth;
        public static Dictionary<int, List<string>> effectLocationMapChains;
        public static Dictionary<int, List<string>> essenceLocationsAtDepth;
        public static Dictionary<int, List<string>> essenceLocationMapChains;
        
        // Variables for file writing.
        public static BinaryWriter targetWriter;
        
        // For compiling list of defineWarps.
        public static StringWriter warpDefinitionList;
        public static string lastWarpMap = "";
        public static int lastWarpEvent = -1;
        public static int lastWarpPage = -1;
        public static string lastWarpSource = "";
        public static string lastWarpDest = "";
        public static int warpRepeatedNameCount = 1;
        public static int warpMultipleCommandsCount = 1;
        
        // Strings used to describe effects and locations (such as for spoiler log).
        public static string[] mapDisplayNames;
        public static Dictionary<string, string> nexusExitNames;
        public static Dictionary<int, int> effectObtainedSwitch;
        public static Dictionary<int, int> effectPossessedSwitch;
        public static Dictionary<int, string> effectProperName;
        public static Dictionary<int, string> effectFilename;
        public static Dictionary<int, string> essenceProperName;
        
        // Variables for keeping spoiler log.
        public static StringWriter spoilerLog;
        public static List<int> itemOrderForLog;
        
        // Database name storage.
        public static string[] heroNames = { };
        public static string[] skillNames = { };
        public static string[] itemNames = { };
        public static string[] monsterNames = { };
        public static string[] troopNames = { };
        public static string[] attributeNames = { };
        public static string[] conditionNames = { };
        public static string[] animationNames = { };
        public static string[] terrainNames = { };
        public static string[] chipSetNames = { };
        public static string[] commonNames = { };
        public static string[] mapNames = { };
        public static string[] switchNames = { };
        public static string[] variableNames = { };
        public static string[] battleCommandNames = { };
        public static string[] classNames = { };
        public static string[] battlerAnimSetNames = { };
        public static string[][] battlerPoseNames = { };
        public static string[][] weaponAnimationNames = { };
        
        // User settings in external file.
        public static Dictionary<string, int> userSettings;
        public static Dictionary<string, string> userSettingsStr;
        #endregion
        
        // Variables for various debug or development purposes.
        public static bool debugMode = false; // Shows debug messages (stack traces for exceptions).
        public static bool showMissingItems = false; // Prints what items were missing for each invalid layout.
        public static bool makeWarpList = false; // After a valid layout is chosen, copies a list of warps to clipboard.
        
        [STAThread]
        static void Main(string[] args)
        {
            UNICODE = Encoding.Unicode;
            
            readEncodings = new Encoding[4];
            readEncodingIDs = new int[4];
            writeEncodings = new Encoding[4];
            writeEncodingIDs = new int[4];
            for (int i = 0; i < 4; i++)
            {
                readEncodings[i] = Encoding.GetEncoding(932);
                readEncodingIDs[i] = 932;
                writeEncodings[i] = Encoding.GetEncoding(932);
                writeEncodingIDs[i] = 932;
            }
            
            loadUserSettings(true);
            
            generateNewSeed();
            
            bool menuShow = true;
            string menuOption;
            while (menuShow)
            {
                loadUserSettings();
                
                Console.Clear();
                Console.WriteLine("=== REM-9: The Yume Nikki Randomizer ver. " + versionStr + " ===\n"
                    + "Enter corresponding number/letter to enter menu or perform action.\n\n"
                    + "1. Difficulty Settings - What to randomize\n"
                    + "2. Goal Settings - How to win\n"
                    + "3. Game Settings - Stuff to tweak\n\n"
                    + "A. Set random seed (Current seed: " + currentSeed + ")\n\n"
                    + "Z. Generate randomized game!\n"
                    + "X. Exit");
                
                menuOption = Console.ReadLine().ToUpper();
                
                switch (menuOption)
                {
                    case "1":
                        difficultySettingsMenu();
                        break;
                    case "2":
                        goalSettingsMenu();
                        break;
                    case "3":
                        gameSettingsMenu();
                        break;
                    case "A":
                        newSeedPrompt();
                        break;
                    case "Z":
                        generateRandomGame();
                        enterToContinue();
                        break;
                    
                    case "X": // Exit
                        menuShow = false;
                        break;
                }
            }
        }
        
        // Presents difficulty options.
        static void difficultySettingsMenu()
        {
            bool menuShow = true;
            int functionListPage = 0;
            string menuOption;
            while (menuShow)
            {
                loadUserSettings();
                
                Console.Clear();
                Console.WriteLine("--- Difficulty Settings ---\n"
                    + "Toggle or modify settings by entering the corresponding number.\n");
                
                if (functionListPage == 0)
                {
                    Console.WriteLine("1. Randomize Effects: " + onOff(randomizeEffects) + "\n"
                        + "Shuffles locations of all Effects.\n\n"
                        + "2. Nexus Locks: " + nexusLockCount + "\n"
                        + "Number of Nexus doors to block off.\n"
                        + "Locks require you obtain a specific Effect. (0 to 11)\n\n"
                        + "3. Entrance Randomization: " + entranceRandoTypes[entranceRandoType] + "\n"
                        + "The type of warp shuffling to perform.\n\n"
                        + "?. Next page\n"
                        + "0. Back");
                }
                else
                {
                    Console.WriteLine("4. Bicycle Placement: " + itemPlacements[bicyclePlacement] + "\n"
                        + "Normal places it normally. Early attempts to place it as early\n"
                        + "and close to the Nexus as possible. Start starts you with it.\n\n"
                        + "5. Eyeball-Hand Placement: " + itemPlacements[eyeballPlacement] + "\n"
                        + "Normal places it normally. Early attempts to place it as early\n"
                        + "and close to the Nexus as possible. Start starts you with it.\n\n"
                        + "6. Nexus Lock Hints: " + onOff(nexusLockHints) + "\n"
                        + "Makes Nexus Locks tell you what area their Effects are in.\n\n"
                        + "7. Generate Spoiler Log: " + onOff(spoilerLogEnabled) + "\n"
                        + "Includes SpoilerLog.txt, which lists item locations and paths.\n\n"
                        + "?. Previous page\n"
                        + "0. Back");
                }
                
                menuOption = Console.ReadLine().ToUpper();
                
                switch (menuOption)
                {
                    case "1":
                        if (functionListPage == 0)
                            randomizeEffects = !randomizeEffects;
                        break;
                    case "2":
                        if (functionListPage == 0)
                            numberPrompt(ref nexusLockCount, 0, 11);
                        break;
                    case "3":
                        if (functionListPage == 0)
                        {
                            Console.WriteLine("- 0. None\n"
                                + "- 1. Nexus Doors (Nexus doors become one-way warps to anywhere)\n"
                                + "- 2. Nexus Doors w/ Return (The way you came goes back to Nexus)\n"
                                + "- 3. Nightmare (Full randomization of dream world)\n"
                                + "- 4. Complete Nightmare (The way you came is a separate warp)");
                            numberPrompt(ref entranceRandoType, 0, 4);
                        }
                        break;
                    case "4":
                        if (functionListPage == 1)
                        {
                            Console.WriteLine("- 0. Normal (Place Bicycle like any other effect)\n"
                                + "- 1. Early (Try to place Bicycle in an early-accessible location)\n"
                                + "- 2. Start With (Start with Bicycle already obtained)");
                            numberPrompt(ref bicyclePlacement, 0, 2);
                        }
                        break;
                    case "5":
                        if (functionListPage == 1)
                        {
                            Console.WriteLine("- 0. Normal (Place Eyeball-hand like any other effect)\n"
                                + "- 1. Early (Try to place Eyeball-hand in an early-accessible location)\n"
                                + "- 2. Start With (Start with Eyeball-hand already obtained)");
                            numberPrompt(ref eyeballPlacement, 0, 2);
                        }
                        break;
                    case "6":
                        if (functionListPage == 1)
                            nexusLockHints = !nexusLockHints;
                        break;
                    case "7":
                        if (functionListPage == 1)
                            spoilerLogEnabled = !spoilerLogEnabled;
                        break;
                    
                    case "?": // See other functions
                    case "/":
                    case " ":
                    case "":
                        functionListPage = (functionListPage + 1) % 2;
                        break;
                    
                    case "0": // Back
                    case "Z":
                        menuShow = false;
                        break;
                }
            }
        }
        
        // Presents goal options.
        static void goalSettingsMenu()
        {
            bool menuShow = true;
            string menuOption;
            while (menuShow)
            {
                loadUserSettings();
                
                Console.Clear();
                Console.WriteLine("--- Goal Settings ---\n"
                    + "Toggle or modify settings by entering the corresponding number.\n");
                
                Console.WriteLine("1. Effects Goal: " + effectsGoal + "\n"
                    + "Number of Effects to place in the Nexus to reach the ending. (1 to 24)\n\n"
                    + "2. Essences Goal: " + essencesGoal + "\n"
                    + "Number of Essences (items from side-events) required for ending. (0 to 8)\n\n"
                    + "3. Minimum Reachable Effects: " + minimumEffects + "\n"
                    + "Minimum of Effects required to be obtainable. (Can't be less than goal)\n\n"
                    + "4. Minimum Reachable Essences: " + minimumEssences + "\n"
                    + "Minimum of Essences required to be obtainable. (Can't be less than goal)\n\n"
                    + "0. Back");
                
                menuOption = Console.ReadLine().ToUpper();
                
                switch (menuOption)
                {
                    case "1":
                        numberPrompt(ref effectsGoal, 1, 24);
                        if (minimumEffects < effectsGoal)
                            minimumEffects = effectsGoal;
                        break;
                    case "2":
                        numberPrompt(ref essencesGoal, 0, 8);
                        if (minimumEssences < essencesGoal)
                            minimumEssences = essencesGoal;
                        break;
                    case "3":
                        numberPrompt(ref minimumEffects, effectsGoal, 24);
                        break;
                    case "4":
                        numberPrompt(ref minimumEssences, essencesGoal, 8);
                        break;
                    
                    case "0": // Back
                    case "Z":
                        menuShow = false;
                        break;
                }
            }
        }
        
        // Presents game options.
        static void gameSettingsMenu()
        {
            bool menuShow = true;
            int functionListPage = 0;
            string menuOption;
            while (menuShow)
            {
                loadUserSettings();
                
                Console.Clear();
                Console.WriteLine("--- Game Settings ---\n"
                    + "Toggle or modify settings by entering the corresponding number.\n");
                
                if (functionListPage == 0)
                {
                    Console.WriteLine("1. Derandomize Events: " + onOff(derandomizeEvents) + "\n"
                        + "Makes changes that remove luck from most random events,\n"
                        + "such as having random chances occur after a set number of times.\n\n"
                        + "2. Fix Chair Glitch: " + onOff(fixChairGlitch) + "\n"
                        + "Removes the ability to open the menu in the dream desk chair.\n"
                        + "If not fixed, you can perform a speed-up glitch using the Bicycle.\n\n"
                        + "3. Walk Speed: " + walkSpeed + "\n"
                        + "Speed of walking. (3 originally, 4 recommended. 1 is slowest, 6 is fastest.)\n\n"
                        + "4. Bike Speed: " + bikeSpeed + "\n"
                        + "Speed of Bicycle. (4 originally, 5 recommended. 1 is slowest, 6 is fastest.)\n\n"
                        + "?. Next page\n"
                        + "0. Back");
                }
                else
                {
                    Console.WriteLine("5. Walk Sound: " + walkSounds[walkSound] + "\n"
                        + "What to use as the basic walk sound effect.\n\n"
                        + "6. Heckification Type: " + heckificationTypes[heckificationType] + "\n"
                        + "Allows you to make the Hell maze easier, or route around it entirely.\n\n"
                        + "7. Easy Teleporter Maze: " + onOff(easyTeleporterMaze) + "\n"
                        + "Has the Teleporter Maze just take you to the end.\n\n"
                        + "8. Darkness Level: " + darknessLevels[darknessLevel] + "\n"
                        + "The darkness of dark areas. Normal is pretty dark, but visible without Lamp.\n"
                        + "Less Dark is easily visible. Pitch Black requires using Lamp to move.\n\n"
                        + "9. Givers Reflect Effect: " + onOff(giversReflectEffect) + "\n"
                        + "Changes Effect-givers to match the new Effect they give.\n\n"
                        + "?. Previous page\n"
                        + "0. Back");
                }
                
                menuOption = Console.ReadLine().ToUpper();
                
                switch (menuOption)
                {
                    case "1":
                        if (functionListPage == 0)
                            derandomizeEvents = !derandomizeEvents;
                        break;
                    case "2":
                        if (functionListPage == 0)
                            fixChairGlitch = !fixChairGlitch;
                        break;
                    case "3":
                        if (functionListPage == 0)
                            numberPrompt(ref walkSpeed, 1, 6);
                        break;
                    case "4":
                        if (functionListPage == 0)
                            numberPrompt(ref bikeSpeed, 1, 6);
                        break;
                    case "5":
                        if (functionListPage == 1)
                        {
                            Console.WriteLine("- 0. Normal\n"
                                + "- 1. Carpet\n"
                                + "- 2. None");
                            numberPrompt(ref walkSound, 0, 2);
                        }
                        break;
                    case "6":
                        if (functionListPage == 1)
                        {
                            Console.WriteLine("- 0. Hell (Keep the same maze layout and warps)\n"
                                + "- 1. Heck (Change maze layout to remove many dead ends)\n"
                                + "- 2. And Goodbye (Make warps into Hell go somewhere out of Hell)");
                            numberPrompt(ref heckificationType, 0, 2);
                        }
                        break;
                    case "7":
                        if (functionListPage == 1)
                            easyTeleporterMaze = !easyTeleporterMaze;
                        break;
                    case "8":
                        if (functionListPage == 1)
                        {
                            Console.WriteLine("- 0. Normal\n"
                                + "- 1. Less Dark\n"
                                + "- 2. Pitch Black (blocked from navigating unless using Lamp)");
                            numberPrompt(ref darknessLevel, 0, 2);
                        }
                        break;
                    case "9":
                        if (functionListPage == 1)
                            giversReflectEffect = !giversReflectEffect;
                        break;
                    
                    case "?": // See other functions
                    case "/":
                    case " ":
                    case "":
                        functionListPage = (functionListPage + 1) % 2;
                        break;
                    
                    case "0": // Back
                    case "Z":
                        menuShow = false;
                        break;
                }
            }
        }
        
        // Prompts user to enter a custom seed.
        static void newSeedPrompt()
        {
            Console.WriteLine("Enter custom seed string. (Leave blank to generate a new one.)");
            currentSeed = Console.ReadLine();
            if (currentSeed.Equals(""))
                generateNewSeed();
        }
        
        // Generates a new random seed from system time.
        static void generateNewSeed()
        {
            currentSeed = new Random((int)DateTime.Now.Ticks).Next().ToString();
        }
        
        // Generates a random game using the current seed.
        static void generateRandomGame()
        {
            Console.WriteLine("Loading base game...");
            if (!loadBaseGame())
                return;
            
            Console.WriteLine("Generating random layout...");
            if (!pickRandomLayout())
                return;
            
            Console.WriteLine("Creating files...");
            saveRandomGame();
            
            generateNewSeed();
        }
       
        // Loads all data from the base game into data structures.
        static bool loadBaseGame()
        {
            gamePath = "Base" + versionID + "\\Data";
            
            if (!Directory.Exists(gamePath))
            {
                Console.WriteLine("Base game (version " + versionStr + ") not found.");
                Console.WriteLine("Check program directory for \"Base" + versionID + "\" folder, containing a \"Data\" folder.");
                return false;
            }
            
            Map startMap = new Map(gamePath + "\\Map0002.lmu");
            if (startMap.baseGameVersion() != versionID)
            {
                Console.WriteLine("Version of base game does not match program version (" + versionStr + ").");
                return false;
            }
            
            mapTree = new MapTree(gamePath + "\\RPG_RT.lmt");
            database = new Database(gamePath + "\\RPG_RT.ldb");
            
            maps = new Dictionary<int, Map>();
            IEnumerable<string> mapList = Directory.EnumerateFiles(gamePath, "*.lmu");
            foreach (string file in mapList)
            {
                string numStr = Path.GetFileNameWithoutExtension(file).Replace("Map", "");
                int mapID = 0;
                if (int.TryParse(numStr, out mapID))
                    maps[mapID] = new Map(file);
            }
            
            if (heckificationType == 1) // Heck layout
            {
                Map heckMap = new Map(gamePath + "\\Map0047b.lmu");
                heckTiles = heckMap.getTileLayer1();
            }
            
            return true;
        }
        
        // Initializes base game locations of all effects.
        static void initBaseEffects()
        {
            effectLocations = new Dictionary<string, CommandLocation>();
            effectLocationText = new Dictionary<string, string>();
            effects = new Dictionary<string, int>();
            
            effectProperName = new Dictionary<int, string>();
            effectFilename = new Dictionary<int, string>();
            effectObtainedSwitch = new Dictionary<int, int>();
            effectPossessedSwitch = new Dictionary<int, int>();
            
            defineEffect("Frog", "Frog", "Forest World / Dense Woods B",
                new CombinedLocation(new MapLocation(MAP_FORESTWORLD, 2), new MapLocation(MAP_JUKAI2, 6)), EF_FROG, 1, 26);
            defineEffect("Umbrella", "Umbrella", "Puddle World",
                new MapLocation(MAP_PUDDLEWORLD, 2), EF_UMBRELLA, 2, 27);
            defineEffect("Hat and scarf", "HatAndScarf", "Block World",
                new MapLocation(MAP_BLOCKWORLD, 2), EF_HATANDSCARF, 3, 28);
            defineEffect("Yuki-onna", "YukiOnna", "Snow World",
                new MapLocation(MAP_SNOWWORLD, 2), EF_YUKIONNA, 4, 29);
            defineEffect("Knife", "Knife", "Dark World",
                new MapLocation(MAP_DARKWORLD, 2), EF_KNIFE, 12, 37);
            defineEffect("Eyeball-hand", "EyeballHand", "Eyeball World",
                new MapLocation(MAP_EYEBALLWORLD, 2), EF_EYEBALLHAND, 5, 30);
            defineEffect("Fat", "Fat", "The Docks (East Side)",
                new MapLocation(MAP_DOCKS, 121, 1, "East Side"), EF_FAT, 6, 31);
            defineEffect("Small", "Small", "Candle World",
                new MapLocation(MAP_CANDLEWORLD, 2), EF_SMALL, 7, 32);
            defineEffect("Flute", "Flute", "The Mall: Flute Room",
                new MapLocation(MAP_MALLMUSIC, 6), EF_FLUTE, 8, 33);
            defineEffect("Neon", "Neon", "Neon World",
                new MapLocation(MAP_NEONWORLD, 3), EF_NEON, 9, 34);
            defineEffect("Nopperabou", "Nopperabou", "Sewers: Processing Plant 2",
                new MapLocation(MAP_GUTTER006, 6), EF_NOPPERABOU, 10, 35);
            defineEffect("Severed head", "SeveredHead", "Guillotine World",
                new CombinedLocation(new MapLocation(MAP_GUILLOTINEWORLDSMALL, 6), new MapLocation(MAP_GUILLOTINEWORLDBIG, 6)), EF_SEVEREDHEAD, 11, 36);
            defineEffect("Towel", "Towel", "The Infinite Wilderness",
                new MapLocation(MAP_INFINITEWILDERNESS, 3), EF_TOWEL, 14, 39);
            defineEffect("Cat", "Cat", "Shield-Folk World",
                new MapLocation(MAP_SHIELDFOLKWORLD, 2), EF_CAT, 15, 40);
            defineEffect("Lamp", "Lamp", "Lamp World",
                new MapLocation(MAP_LAMPWORLD, 3), EF_LAMP, 16, 41);
            defineEffect("Bicycle", "Bicycle", "Graffiti World",
                new MapLocation(MAP_GRAFFITIWORLD, 2), EF_BICYCLE, 17, 42);
            defineEffect("Long hair", "LongHair", "Mural World: Long Hair",
                new MapLocation(MAP_MURALWORLD, 2), EF_LONGHAIR, 18, 43);
            defineEffect("Poop hair", "PoopHair", "Downstairs Storage Room",
                new MapLocation(MAP_STORAGEROOM, 29), EF_POOPHAIR, 19, 44);
            defineEffect("Blonde hair", "BlondeHair", "Mural World: Blonde Hair",
                new MapLocation(MAP_MURALWORLD, 4), EF_BLONDEHAIR, 20, 45);
            defineEffect("Triangle kerchief", "TriangleKerchief", "Ghost World",
                new MapLocation(MAP_GHOSTWORLD, 14), EF_TRIANGLEKERCHIEF, 13, 38);
            defineEffect("Witch", "Witch", "Dark Woods: Witch's Island",
                new MapLocation(MAP_WITCHISLAND, 4), EF_WITCH, 21, 46);
            defineEffect("Oni", "Oni", "FC World A: Basement",
                new MapLocation(MAP_FCBASEMENT019, 3), EF_ONI, 22, 47);
            defineEffect("Squishy", "Squishy", "FC World B: Under-Village",
                new MapLocation(MAP_FCUNDERVILLAGE, 2), EF_SQUISHY, 23, 48);
            defineEffect("Stoplight", "Stoplight", "Dense Woods B: The Road",
                new MapLocation(MAP_JUKAI2, 18, 1, "Road"), EF_STOPLIGHT, 24, 49);
        }
        
        // Defines an effect location, an effect ID (to be shuffled around), and info about the effect.
        static void defineEffect(string properName, string filename, string locationText, CommandLocation location, int effect, int obtainedSwitch, int possessedSwitch)
        {
            effectLocations[properName] = location;
            effectLocationText[properName] = locationText;
            effects[properName] = effect;
            
            effectProperName[effect] = properName;
            effectFilename[effect] = filename;
            effectObtainedSwitch[effect] = obtainedSwitch;
            effectPossessedSwitch[effect] = possessedSwitch;
        }
        
        // Initializes base game locations of all essences.
        static void initBaseEssences()
        {
            essenceLocations = new Dictionary<string, CommandLocation>();
            essences = new Dictionary<string, int>();
            
            essenceProperName = new Dictionary<int, string>();
            
            defineEssence("Essence of Uboa", new MapLocation(MAP_PONIKOHOUSE, 3, 5), ES_UBOA);
            defineEssence("Essence of Monoe", new MapLocation(MAP_MONO003, 1), ES_MONOE);
            defineEssence("Essence of Monoko", new MapLocation(MAP_MONO001, 1, 4), ES_MONOKO); // Requires Stoplight
            defineEssence("Essence of Masada", new MapLocation(MAP_SPACESHIP01, 20), ES_MASADA);
            defineEssence("Essence of Face", new MapLocation(MAP_FACE, 1, 2), ES_FACE);
            defineEssence("Essence of Flight", new MapLocation(MAP_WITCHFLIGHT, 5), ES_FLIGHT);
            defineEssence("Essence of Party", new MapLocation(MAP_WILDERNESSPARTY, 8), ES_PARTY);
            defineEssence("Essence of Nasu", new MapLocation(MAP_NASU, 11, 3), ES_NASU);
        }
        
        // Defines an essence location, an essence ID, and info about the essence.
        static void defineEssence(string properName, CommandLocation location, int essence)
        {
            essenceLocations[properName] = location;
            essences[properName] = essence;
            
            essenceProperName[essence] = properName;
        }
        
        // Initializes base game locations of all warp locations and where they go.
        static void initBaseWarps()
        {
            warpLocations = new Dictionary<string, CommandLocation>();
            exits = new Dictionary<string, Warp>();
            
            noShuffleKeys = new List<string>();
            
            // Common Events
            defineWarp("EyeballHandCommon/DoorRoom", new CommonLocation(70), new Warp(MAP_NEXUS, 17, 15, D_SAME), false); // Eyeball-hand back to Nexus
            defineWarp("WakeUpCommon-Normal/RealWorldRoom-Normal", new CommonLocation(94, 2), new Warp(MAP_REALWORLDROOM, 13, 7, D_SAME), false); // Normal waking up
            defineWarp("WakeUpCommon-Fall/RealWorldRoom-Fall", new CommonLocation(94), new Warp(MAP_REALWORLDROOM, 12, 7, D_SAME), false); // Fall out of bed
            defineWarp("DreamBedCommon/StairsInBed", new CommonLocation(122), new Warp(MAP_STAIRCASEOFHANDS, 54, 8, D_LEFT)); // Random dream bed warp
            
            // Real World
            defineWarp("GameStart/RealWorldRoom", new MapLocation(MAP_GAMESTART, 1, 5), new Warp(MAP_REALWORLDROOM, 10, 9, D_DOWN), false);
            
            defineWarp("RealWorldRoom/Veranda", new CombinedLocation(new MapLocation(MAP_REALWORLDROOM, 5, 1), new MapLocation(MAP_REALWORLDROOM, 9, 1)),
                                                new Warp(MAP_VERANDA, 16, 7, D_DOWN), false);
            defineWarp("RealWorldRoom-SleepA/DreamVerandaA", new MapLocation(MAP_REALWORLDROOM, 13, 1, 2), new Warp(MAP_DREAMVERANDAA, 16, 8, D_DOWN), false);
            defineWarp("RealWorldRoom-SleepB/DreamVerandaB", new MapLocation(MAP_REALWORLDROOM, 13, 1), new Warp(MAP_DREAMVERANDAB, 16, 8, D_DOWN), false);
            defineWarp("RealWorldRoom/FamiGame1", new MapLocation(MAP_REALWORLDROOM, 18, 1), new Warp(MAP_NASU, 0, 0, D_SAME), false);
            
            defineWarp("Veranda/RealWorldRoom", new MapLocation(MAP_VERANDA, 7, 1), new Warp(MAP_REALWORLDROOM, 10, 12, D_UP), false);
            defineWarp("Veranda/Ending", new MapLocation(MAP_VERANDA, 26, 1), new Warp(MAP_ENDING, 20, 20, D_SAME), false);
            
            defineWarp("FamiGame1/RealWorldRoom", new MapLocation(MAP_NASU, 2, 2), new Warp(MAP_REALWORLDROOM, 7, 7, D_LEFT), false);
            
            // Starting Dream Rooms
            defineWarp("DreamRoom-OutsideA/DreamVerandaA", new CombinedLocation(new MapLocation(MAP_DREAMROOM, 4, 1, 2), new MapLocation(MAP_DREAMROOM, 5, 1, 2)),
                                                           new Warp(MAP_DREAMVERANDAA, 16, 7, D_DOWN), false);
            defineWarp("DreamRoom-OutsideB/DreamVerandaB", new CombinedLocation(new MapLocation(MAP_DREAMROOM, 4, 1), new MapLocation(MAP_DREAMROOM, 5, 1)),
                                                           new Warp(MAP_DREAMVERANDAB, 16, 7, D_DOWN), false);
            defineWarp("DreamRoom/DoorRoom", new MapLocation(MAP_DREAMROOM, 12, 1), new Warp(MAP_NEXUS, 17, 5, D_DOWN), false);
            
            defineWarp("DreamVerandaA/DreamRoom-OutsideA", new MapLocation(MAP_DREAMVERANDAA, 11, 1), new Warp(MAP_DREAMROOM, 10, 12, D_UP), false);
            
            defineWarp("DreamVerandaB/DreamRoom-OutsideB", new MapLocation(MAP_DREAMVERANDAB, 8, 1), new Warp(MAP_DREAMROOM, 10, 12, D_UP), false);
            
            defineWarp("DoorRoom/DreamRoom", new MapLocation(MAP_NEXUS, 1, 1), new Warp(MAP_DREAMROOM, 7, 6, D_DOWN), false);
            defineWarp("DoorRoom/01Frog", new MapLocation(MAP_NEXUS, 3, 1), new Warp(MAP_FORESTWORLD, 53, 57, D_DOWN));
            defineWarp("DoorRoom/02HatAndScarf", new MapLocation(MAP_NEXUS, 2, 1), new Warp(MAP_BLOCKWORLD, 12, 11, D_DOWN));
            defineWarp("DoorRoom/03Umbrella", new MapLocation(MAP_NEXUS, 6, 1), new Warp(MAP_PUDDLEWORLD, 11, 11, D_DOWN));
            defineWarp("DoorRoom/04Knife", new MapLocation(MAP_NEXUS, 4, 1), new Warp(MAP_DARKWORLD, 12, 11, D_DOWN));
            defineWarp("DoorRoom/05YukiOnna", new MapLocation(MAP_NEXUS, 7, 1), new Warp(MAP_SNOWWORLD, 10, 11, D_DOWN));
            defineWarp("DoorRoom/06Hairstyle", new MapLocation(MAP_NEXUS, 8, 1), new Warp(MAP_MURALWORLD, 13, 12, D_DOWN));
            defineWarp("DoorRoom/07Bicycle", new MapLocation(MAP_NEXUS, 9, 1), new Warp(MAP_GRAFFITIWORLD, 12, 12, D_DOWN));
            defineWarp("DoorRoom/08EyeballHand", new MapLocation(MAP_NEXUS, 10, 1), new Warp(MAP_EYEBALLWORLD, 11, 11, D_DOWN));
            defineWarp("DoorRoom/09Small", new MapLocation(MAP_NEXUS, 11, 1), new Warp(MAP_CANDLEWORLD, 12, 11, D_DOWN));
            defineWarp("DoorRoom/10CatEars", new MapLocation(MAP_NEXUS, 12, 1), new Warp(MAP_SHIELDFOLKWORLD, 13, 13, D_DOWN));
            defineWarp("DoorRoom/11Neon", new MapLocation(MAP_NEXUS, 13, 1), new Warp(MAP_NEONWORLD, 12, 11, D_DOWN));
            defineWarp("DoorRoom/12Lamp", new MapLocation(MAP_NEXUS, 5, 1), new Warp(MAP_NUMBERWORLD, 83, 30, D_DOWN));
            
            // 01 Frog
            defineWarp("01Frog/DoorRoom", new MapLocation(MAP_FORESTWORLD, 1, 1), new Warp(MAP_NEXUS, 15, 8, D_DOWN));
            defineWarp("01Frog-GateUp/Thorns-Bottom", new MapLocation(MAP_FORESTWORLD, 3, 1), new Warp(MAP_FACECARPETPLAZA, 29, 42, D_UP));
            defineWarp("01Frog-GateDown/Thorns-Top", new MapLocation(MAP_FORESTWORLD, 3, 1, 2), new Warp(MAP_FACECARPETPLAZA, 20, 7, D_DOWN));
            
            defineWarp("Thorns-Top/01Frog-GateDown", new MapLocation(MAP_FACECARPETPLAZA, 1, 1), new Warp(MAP_FORESTWORLD, 93, 22, D_UP));
            defineWarp("Thorns-Bottom/01Frog-GateUp", new MapLocation(MAP_FACECARPETPLAZA, 2, 1), new Warp(MAP_FORESTWORLD, 93, 22, D_DOWN));
            defineWarp("Thorns-RedBoxes/RedA-Thorns", new CombinedLocation(new MapLocation(MAP_FACECARPETPLAZA, 4, 1), new MapLocation(MAP_FACECARPETPLAZA, 4, 4)),
                                                      new Warp(MAP_HELL, 94, 61, D_DOWN));
            defineWarp("Thorns-DressToriningen/NeonTilesA-Stuck", new MapLocation(MAP_FACECARPETPLAZA, 3, 6), new Warp(MAP_NEONTILESA, 11, 43, 2, "Stuck"), false);
            
            // 02 Hat and Scarf
            defineWarp("02HatAndScarf/DoorRoom", new MapLocation(MAP_BLOCKWORLD, 1, 1), new Warp(MAP_NEXUS, 11, 10, D_DOWN));
            defineWarp("02HatAndScarf-Gate/Mono-Mouth", new MapLocation(MAP_BLOCKWORLD, 23, 2), new Warp(MAP_MONO, 34, 34, D_DOWN));
            // Also has Mafurako warps within same map
            
            // 03 Umbrella
            defineWarp("03Umbrella/DoorRoom", new MapLocation(MAP_PUDDLEWORLD, 1, 1), new Warp(MAP_NEXUS, 9, 14, D_DOWN));
            defineWarp("03Umbrella-Subway/Highway-Subway", new MapLocation(MAP_PUDDLEWORLD, 3, 1), new Warp(MAP_HIGHWAY, 130, 30, D_LEFT));
            
            defineWarp("Highway-Subway/03Umbrella-Subway", new MapLocation(MAP_HIGHWAY, 34, 1), new Warp(MAP_PUDDLEWORLD, 72, 53, D_LEFT));
            
            /* Infinite Road Maps: Best not to randomize, so walking along it works as normal */
            defineWarp("Highway-RightTop/HighwayA-LeftTop", new MapLocation(MAP_HIGHWAY, 78, 2), new Warp(MAP_HIGHWAYA, 15, 34, D_SAME), false);
            defineWarp("Highway-RightBottom/HighwayA-LeftBottom", new MapLocation(MAP_HIGHWAY, 79, 2), new Warp(MAP_HIGHWAYA, 15, 35, D_SAME), false);
            
            defineWarp("HighwayA-LeftTop/Highway-RightTop", new MapLocation(MAP_HIGHWAYA, 2, 1), new Warp(MAP_HIGHWAY, 16, 34, D_SAME), false);
            defineWarp("HighwayA-LeftBottom/Highway-RightBottom", new MapLocation(MAP_HIGHWAYA, 3, 1), new Warp(MAP_HIGHWAY, 16, 35, D_SAME), false);
            defineWarp("HighwayA-RightTop/HighwayB-LeftTop", new MapLocation(MAP_HIGHWAYA, 1, 1), new Warp(MAP_HIGHWAYB, 0, 15, D_RIGHT), false);
            defineWarp("HighwayA-RightBottom/HighwayB-LeftBottom", new MapLocation(MAP_HIGHWAYA, 4, 1), new Warp(MAP_HIGHWAYB, 0, 16, D_RIGHT), false);
            
            defineWarp("HighwayB-LeftTop/HighwayA-RightTop", new MapLocation(MAP_HIGHWAYB, 3, 1), new Warp(MAP_HIGHWAYA, 39, 34, D_LEFT), false);
            defineWarp("HighwayB-LeftBottom/HighwayA-RightBottom", new MapLocation(MAP_HIGHWAYB, 4, 1), new Warp(MAP_HIGHWAYA, 39, 35, D_LEFT), false);
            defineWarp("HighwayB-RightTop/HighwayC-LeftTop", new MapLocation(MAP_HIGHWAYB, 1, 1), new Warp(MAP_HIGHWAYC, 1, 15, D_RIGHT), false);
            defineWarp("HighwayB-RightBottom/HighwayC-LeftBottom", new MapLocation(MAP_HIGHWAYB, 2, 1), new Warp(MAP_HIGHWAYC, 1, 16, D_RIGHT), false);
            
            defineWarp("HighwayC-RightTop/HighwayD-LeftTop", new MapLocation(MAP_HIGHWAYC, 1, 2), new Warp(MAP_HIGHWAYD, 18, 15, D_SAME), false);
            defineWarp("HighwayC-RightBottom/HighwayD-LeftBottom", new MapLocation(MAP_HIGHWAYC, 2, 2), new Warp(MAP_HIGHWAYD, 18, 16, D_SAME), false);
            
            defineWarp("HighwayD-RightTop/HighwayC-RightTop", new MapLocation(MAP_HIGHWAYD, 1, 1), new Warp(MAP_HIGHWAYC, 0, 15, D_RIGHT), false);
            defineWarp("HighwayD-RightBottom/HighwayC-RightBottom", new MapLocation(MAP_HIGHWAYD, 2, 1), new Warp(MAP_HIGHWAYC, 0, 16, D_RIGHT), false);
            defineWarp("HighwayD-LeftTop/Jukai2-RoadTop", new MapLocation(MAP_HIGHWAYD, 3, 1), new Warp(MAP_JUKAI2, 79, 38, 3, "Road"), false);
            defineWarp("HighwayD-LeftBottom/Jukai2-RoadBottom", new MapLocation(MAP_HIGHWAYD, 4, 1), new Warp(MAP_JUKAI2, 79, 39, 3, "Road"), false);
            /* End of Infinite Road maps */
            
            // 04 Knife
            defineWarp("04Knife/DoorRoom", new MapLocation(MAP_DARKWORLD, 1, 1), new Warp(MAP_NEXUS, 9, 18, D_DOWN));
            defineWarp("04Knife-Gate/Sand-Gate", new MapLocation(MAP_DARKWORLD, 7, 1), new Warp(MAP_WILDERNESSGATE, 18, 12, D_DOWN));
            
            defineWarp("Sand-Gate/04Knife-Gate", new MapLocation(MAP_WILDERNESSGATE, 4, 1), new Warp(MAP_DARKWORLD, 49, 72, D_DOWN));
            defineWarp("Sand-Left/SandH-Right", new MapLocation(MAP_WILDERNESSGATE, 1, 1), new Warp(MAP_WILDERNESSROUTE3, 49, 36, D_LEFT));
            defineWarp("Sand-Right/SandB-Left", new CombinedLocation(new MapLocation(MAP_WILDERNESSGATE, 2, 1), new MapLocation(MAP_WILDERNESSGATE, 3, 1)),
                                                new Warp(MAP_WILDERNESSROUTE1, 0, 15, D_RIGHT));
            
            defineWarp("SandA-Right/SandH-Left", new MapLocation(MAP_WILDERNESSROUTE4, 1, 1), new Warp(MAP_WILDERNESSROUTE3, 0, 14, D_RIGHT));
            defineWarp("SandA-Left/SandE-Party", new CombinedLocation(new MapLocation(MAP_WILDERNESSROUTE4, 2, 1), new MapLocation(MAP_WILDERNESSROUTE4, 3, 1)),
                                                 new Warp(MAP_WILDERNESSPARTY, 29, 12, D_LEFT)); // Merged Y=12/Y=13 warps
            
            defineWarp("SandB-Left/Sand-Right", new MapLocation(MAP_WILDERNESSROUTE1, 1, 1), new Warp(MAP_WILDERNESSGATE, 29, 13, D_LEFT));
            defineWarp("SandB-UpLeft/SandF-Down", new MapLocation(MAP_WILDERNESSROUTE1, 3, 1), new Warp(MAP_WILDERNESSTOSETTLEMENT, 51, 59, D_UP));
            defineWarp("SandB-UpRight/SandC-Down", new MapLocation(MAP_WILDERNESSROUTE1, 2, 1), new Warp(MAP_WILDERNESSROUTE2, 18, 29, D_UP));
            
            defineWarp("SandC-Down/SandB-UpRight", new MapLocation(MAP_WILDERNESSROUTE2, 1, 1), new Warp(MAP_WILDERNESSROUTE1, 23, 0, D_DOWN));
            defineWarp("SandC-UpLeft/SandG-Down", new MapLocation(MAP_WILDERNESSROUTE2, 3, 1), new Warp(MAP_WILDERNESSFCPORTAL, 14, 29, 0, "Front"));
            defineWarp("SandC-UpRight/SandD-Down", new MapLocation(MAP_WILDERNESSROUTE2, 2, 1), new Warp(MAP_WILDERNESSRAVEBOX, 18, 29, D_UP));
            
            defineWarp("SandD-Down/SandC-UpRight", new MapLocation(MAP_WILDERNESSRAVEBOX, 1, 1), new Warp(MAP_WILDERNESSROUTE2, 23, 0, D_DOWN));
            
            defineWarp("SandE-Party/SandA-Left", new CombinedLocation(new MapLocation(MAP_WILDERNESSPARTY, 1, 1), new MapLocation(MAP_WILDERNESSPARTY, 3, 1)),
                                                 new Warp(MAP_WILDERNESSROUTE4, 0, 17, D_RIGHT)); // Merged Y=17/Y=18 warps
            
            defineWarp("SandF-Down/SandB-UpLeft", new MapLocation(MAP_WILDERNESSTOSETTLEMENT, 2, 1), new Warp(MAP_WILDERNESSROUTE1, 12, 0, D_DOWN));
            defineWarp("SandF-Fence/Earth-Fence", new MapLocation(MAP_WILDERNESSTOSETTLEMENT, 1, 1), new Warp(MAP_BARRACKSSETTLEMENT, 30, 58, D_UP));
            
            defineWarp("SandG-Box/FCEntranceB", new MapLocation(MAP_WILDERNESSFCPORTAL, 1, 1, "Behind Fence"), new Warp(MAP_FCENTRANCEB, 9, 14, D_UP));
            defineWarp("SandG-Down/SandC-UpLeft", new MapLocation(MAP_WILDERNESSFCPORTAL, 4, 1, "Front"), new Warp(MAP_WILDERNESSROUTE2, 9, 0, D_DOWN));
            defineWarp("SandG-Left/SandK-Right", new MapLocation(MAP_WILDERNESSFCPORTAL, 17, 1, "Bottom-Left"), new Warp(MAP_WILDERNESSFENCE, 29, 25, D_LEFT));
            
            defineWarp("FCEntranceB/SandG-Box", new MapLocation(MAP_FCENTRANCEB, 3, 1), new Warp(MAP_WILDERNESSFCPORTAL, 15, 12, 2, "Behind Fence"));
            defineWarp("FCEntranceB/FCSmallMap001-Cave", new MapLocation(MAP_FCENTRANCEB, 2, 1), new Warp(MAP_FCCAVEMOUTH, 9, 7, D_DOWN));
            
            defineWarp("SandH-Right/Sand-Left", new MapLocation(MAP_WILDERNESSROUTE3, 1, 1), new Warp(MAP_WILDERNESSGATE, 0, 15, D_RIGHT));
            defineWarp("SandH-Left/SandA-Right", new CombinedLocation(new MapLocation(MAP_WILDERNESSROUTE3, 2, 1), new MapLocation(MAP_WILDERNESSROUTE3, 3, 1)),
                                                 new Warp(MAP_WILDERNESSROUTE4, 29, 16, D_LEFT));
            defineWarp("SandH-Passage/SandJ-Passage", new MapLocation(MAP_WILDERNESSROUTE3, 4, 1), new Warp(MAP_INFINITEWILDERNESS, 51, 60, D_UP), false); // Instant warp, best not to randomize
            
            defineWarp("SandI-Up/SandL-Down", new CombinedLocation(new MapLocation(MAP_WILDERNESSGAPINFENCE, 3, 1), new MapLocation(MAP_WILDERNESSGAPINFENCE, 14, 1)),
                                              new Warp(MAP_WILDERNESSPURPLEPYLONS, 14, 29, D_UP)); // Merged X=14/X=15 warps
            defineWarp("SandI-Right/SandK-Left", new MapLocation(MAP_WILDERNESSGAPINFENCE, 36, 1), new Warp(MAP_WILDERNESSFENCE, 0, 25, D_RIGHT));
            defineWarp("SandI-Passage/SandJ-Passage", new MapLocation(MAP_WILDERNESSGAPINFENCE, 2, 1), new Warp(MAP_INFINITEWILDERNESS, 138, 28, D_DOWN), false); // Instant warp, best not to randomize
            
            defineWarp("SandJ-Shack/InsideShackA", new MapLocation(MAP_INFINITEWILDERNESS, 1, 1), new Warp(MAP_INSIDESHACKA, 29, 17, D_LEFT));
            defineWarp("SandJ-Passage/SandH-Passage", new MapLocation(MAP_INFINITEWILDERNESS, 4, 1), new Warp(MAP_WILDERNESSROUTE3, 35, 14, D_DOWN), false); // Instant warp, best not to randomize
            defineWarp("SandJ-Passage/SandI-Passage", new MapLocation(MAP_INFINITEWILDERNESS, 2, 1), new Warp(MAP_WILDERNESSGAPINFENCE, 34, 53, D_UP), false); // Instant warp, best not to randomize
            
            defineWarp("InsideShackA/SandJ-Shack", new MapLocation(MAP_INSIDESHACKA, 1, 1), new Warp(MAP_INFINITEWILDERNESS, 98, 122, D_RIGHT));
            
            defineWarp("SandK-Right/SandG-Left", new MapLocation(MAP_WILDERNESSFENCE, 4, 1), new Warp(MAP_WILDERNESSFCPORTAL, 0, 25, 1, "Bottom-Left"));
            defineWarp("SandK-Left/SandI-Right", new MapLocation(MAP_WILDERNESSFENCE, 17, 1), new Warp(MAP_WILDERNESSGAPINFENCE, 69, 32, D_LEFT));
            
            defineWarp("SandL-Down/SandI-Up", new CombinedLocation(new MapLocation(MAP_WILDERNESSPURPLEPYLONS, 1, 1), new MapLocation(MAP_WILDERNESSPURPLEPYLONS, 3, 1)),
                                              new Warp(MAP_WILDERNESSGAPINFENCE, 50, 0, D_DOWN)); // Merged X=50/Y=51 warps
            defineWarp("SandL-Right/SandM-Left", new CombinedLocation(new MapLocation(MAP_WILDERNESSPURPLEPYLONS, 2, 1), new MapLocation(MAP_WILDERNESSPURPLEPYLONS, 4, 1)),
                                                 new Warp(MAP_WILDERNESSFOOTOFSTAIRS, 0, 14, D_RIGHT)); // Merged Y=14/Y=15 warps
            
            defineWarp("SandM-Stairs/SandStairs", new MapLocation(MAP_WILDERNESSFOOTOFSTAIRS, 4, 1), new Warp(MAP_STAIRWAYTOTHESKY, 12, 198, D_RIGHT));
            defineWarp("SandM-Left/SandL-Right", new CombinedLocation(new MapLocation(MAP_WILDERNESSFOOTOFSTAIRS, 9, 1), new MapLocation(MAP_WILDERNESSFOOTOFSTAIRS, 26, 1)),
                                                 new Warp(MAP_WILDERNESSPURPLEPYLONS, 29, 6, D_LEFT)); // Merged Y=6/Y=7 warps
            
            defineWarp("SandStairs/SandM-Stairs", new MapLocation(MAP_STAIRWAYTOTHESKY, 12, 1), new Warp(MAP_WILDERNESSFOOTOFSTAIRS, 48, 13, D_LEFT));
            defineWarp("SandStairs/Park-Stairs", new MapLocation(MAP_STAIRWAYTOTHESKY, 274, 1), new Warp(MAP_PARK, 0, 26, D_RIGHT));
            
            // 05 Yuki-Onna
            defineWarp("05YukiOnna/DoorRoom", new MapLocation(MAP_SNOWWORLD, 1, 1), new Warp(MAP_NEXUS, 11, 22, D_DOWN));
            defineWarp("05YukiOnna/Igloo1", new MapLocation(MAP_SNOWWORLD, 10, 1), new Warp(MAP_IGLOO1, 8, 8, D_RIGHT));
            defineWarp("05YukiOnna/Igloo2", new MapLocation(MAP_SNOWWORLD, 11, 1), new Warp(MAP_IGLOO2, 8, 8, D_RIGHT));
            defineWarp("05YukiOnna/Igloo3", new MapLocation(MAP_SNOWWORLD, 12, 1), new Warp(MAP_IGLOO3, 8, 8, D_RIGHT));
            defineWarp("05YukiOnna/Igloo4", new MapLocation(MAP_SNOWWORLD, 13, 1), new Warp(MAP_IGLOO4, 8, 8, D_RIGHT));
            defineWarp("05YukiOnna/Igloo5", new MapLocation(MAP_SNOWWORLD, 14, 1), new Warp(MAP_IGLOO5, 8, 8, D_RIGHT));
            defineWarp("05YukiOnna/Igloo6", new MapLocation(MAP_SNOWWORLD, 15, 1), new Warp(MAP_IGLOO6, 8, 8, D_RIGHT));
            defineWarp("05YukiOnna/Igloo7", new MapLocation(MAP_SNOWWORLD, 4, 1), new Warp(MAP_IGLOO7, 8, 8, D_RIGHT));
            defineWarp("05YukiOnna-Toriningen/RedA-Stuck", new MapLocation(MAP_SNOWWORLD, 8, 11), new Warp(MAP_HELL, 77, 83, 2, "Stuck"), false); // Need to stab first
            
            defineWarp("Igloo1/05YukiOnna", new MapLocation(MAP_IGLOO1, 1, 1), new Warp(MAP_SNOWWORLD, 39, 34, D_LEFT));
            
            defineWarp("Igloo2/05YukiOnna", new MapLocation(MAP_IGLOO2, 1, 1), new Warp(MAP_SNOWWORLD, 32, 37, D_LEFT));
            
            defineWarp("Igloo3/05YukiOnna", new MapLocation(MAP_IGLOO3, 1, 1), new Warp(MAP_SNOWWORLD, 48, 35, D_LEFT));
            defineWarp("Igloo3-Toriningen/RedA-Stuck", new MapLocation(MAP_IGLOO3, 4, 6), new Warp(MAP_HELL, 77, 83, 2, "Stuck"), false); // Same as one outside but without stab
            
            defineWarp("Igloo4/05YukiOnna", new MapLocation(MAP_IGLOO4, 1, 1), new Warp(MAP_SNOWWORLD, 37, 43, D_LEFT));
            
            defineWarp("Igloo5/05YukiOnna", new MapLocation(MAP_IGLOO5, 1, 1), new Warp(MAP_SNOWWORLD, 45, 41, D_LEFT));
            
            defineWarp("Igloo6/05YukiOnna", new MapLocation(MAP_IGLOO6, 1, 1), new Warp(MAP_SNOWWORLD, 22, 66, D_LEFT));
            
            defineWarp("Igloo7/05YukiOnna", new MapLocation(MAP_IGLOO7, 1, 1), new Warp(MAP_SNOWWORLD, 83, 87, D_LEFT));
            defineWarp("Igloo7-Pond/Shallows", new MapLocation(MAP_IGLOO7, 2, 1), new Warp(MAP_PINKSEA, 23, 15, D_DOWN));
            
            // 06 Hairstyle
            defineWarp("06Hairstyle/DoorRoom", new MapLocation(MAP_MURALWORLD, 1, 1), new Warp(MAP_NEXUS, 15, 24, D_DOWN));
            defineWarp("06Hairstyle-Manhole/Gutter_001-Ladder", new MapLocation(MAP_MURALWORLD, 6, 1), new Warp(MAP_GUTTER001, 68, 12, D_DOWN));
            
            defineWarp("Gutter_001-Ladder/06Hairstyle-Manhole", new MapLocation(MAP_GUTTER001, 4, 1), new Warp(MAP_MURALWORLD, 102, 52, D_DOWN));
            defineWarp("Gutter_001-Left/Gutter_002-Right", new MapLocation(MAP_GUTTER001, 3, 1), new Warp(MAP_GUTTER002, 129, 13, D_LEFT));
            
            defineWarp("Gutter_002-Right/Gutter_001-Left", new MapLocation(MAP_GUTTER002, 2, 1), new Warp(MAP_GUTTER001, 13, 13, D_RIGHT));
            defineWarp("Gutter_002-Left/Gutter_003-Right", new MapLocation(MAP_GUTTER002, 6, 1), new Warp(MAP_GUTTER003, 50, 13, D_LEFT));
            
            defineWarp("Gutter_003-Right/Gutter_002-Left", new MapLocation(MAP_GUTTER003, 1, 1), new Warp(MAP_GUTTER002, 7, 13, D_RIGHT));
            defineWarp("Gutter_003-Left/Gutter_006-Right", new MapLocation(MAP_GUTTER003, 3, 1), new Warp(MAP_GUTTER006, 45, 20, D_LEFT));
            defineWarp("Gutter_003-Up/Gutter_004-Left", new MapLocation(MAP_GUTTER003, 5, 1), new Warp(MAP_GUTTER004, 0, 14, D_RIGHT));
            
            defineWarp("Gutter_004-Left/Gutter_003-Up", new MapLocation(MAP_GUTTER004, 6, 1), new Warp(MAP_GUTTER003, 30, 11, D_DOWN));
            defineWarp("Gutter_004-Right/RedRoom", new MapLocation(MAP_GUTTER004, 8, 1), new Warp(MAP_BIGRED, 0, 13, D_RIGHT));
            defineWarp("Gutter_004-SmallOpening/Gutter_005-Left", new MapLocation(MAP_GUTTER004, 9, 1), new Warp(MAP_GUTTER005, 1, 14, D_RIGHT)); // Requires Small
            
            defineWarp("Gutter_005-Left/Gutter_004-SmallOpening", new MapLocation(MAP_GUTTER005, 6, 1), new Warp(MAP_GUTTER004, 45, 14, D_DOWN)); // Does not require Small
            defineWarp("Gutter_005-Stairs/LakeCorridor-ByVending", new MapLocation(MAP_GUTTER005, 1, 1), new Warp(MAP_DOCKS, 70, 103, 1, "West Side"));
            
            defineWarp("Gutter_006-Right/Gutter_003-Left", new MapLocation(MAP_GUTTER006, 2, 1), new Warp(MAP_GUTTER003, 13, 13, D_RIGHT));
            
            defineWarp("RedRoom/Gutter_004-Right", new MapLocation(MAP_BIGRED, 4, 1), new Warp(MAP_GUTTER004, 79, 14, D_LEFT));
            defineWarp("RedRoom-Mouth/TatamiCave", new MapLocation(MAP_BIGRED, 1, 1), new Warp(MAP_WINDMILLWORLD, 57, 15, D_DOWN));
            
            defineWarp("TatamiCave-Hedge/02HatAndScarf-HedgeExit", new MapLocation(MAP_WINDMILLWORLD, 3, 1), new Warp(MAP_BLOCKWORLD, 110, 57, D_DOWN));
            defineWarp("TatamiCave-Fisherman/LakeCorridor-TopLeft", new MapLocation(MAP_WINDMILLWORLD, 11, 1), new Warp(MAP_DOCKS, 5, 8, 2, "West Side"));
            
            // 07 Bicycle
            defineWarp("07Bicycle/DoorRoom", new MapLocation(MAP_GRAFFITIWORLD, 4, 1), new Warp(MAP_NEXUS, 19, 24, D_DOWN));
            defineWarp("07Bicycle-Elevator/DepartmentA-Elevator", new MapLocation(MAP_GRAFFITIWORLD, 7, 4), new Warp(MAP_MALLA, 4, 6, D_DOWN));
            
            defineWarp("DepartmentA-Elevator/07Bicycle-Elevator", new MapLocation(MAP_MALLA, 8, 4), new Warp(MAP_GRAFFITIWORLD, 129, 84, D_DOWN));
            defineWarp("DepartmentA-DownEscalator/DepartmentB-UpEscalator", new MapLocation(MAP_MALLA, 7, 1), new Warp(MAP_MALLB, 51, 41, 1, true));
            defineWarp("DepartmentA-UpEscalator/DepartmentD-Escalator", new MapLocation(MAP_MALLA, 3, 1), new Warp(MAP_MALLD, 6, 8, D_RIGHT));
            
            defineWarp("DepartmentB-UpEscalator/DepartmentA-DownEscalator", new MapLocation(MAP_MALLB, 12, 1), new Warp(MAP_MALLA, 14, 8, 3, true));
            defineWarp("DepartmentB-DownEscalator/DepartmentC-UpEscalator", new MapLocation(MAP_MALLB, 14, 1), new Warp(MAP_MALLC, 15, 9, 3, true));
            defineWarp("DepartmentB-LeftDoor/EntranceA", new MapLocation(MAP_MALLB, 10, 1), new Warp(MAP_MALLRECEPTION, 17, 9, D_LEFT));
            defineWarp("DepartmentB-RightDoor/EntranceC", new MapLocation(MAP_MALLB, 1, 1), new Warp(MAP_MALLTOKUTO, 4, 9, D_RIGHT));
            
            defineWarp("DepartmentC-UpEscalator/DepartmentB-DownEscalator", new MapLocation(MAP_MALLC, 9, 1), new Warp(MAP_MALLB, 50, 42, 1, true));
            defineWarp("DepartmentC-Hole/Jukai1-Hole", new CombinedLocation(new MapLocation(MAP_MALLC, 2, 1), new MapLocation(MAP_MALLC, 4, 1)),
                                                       new Warp(MAP_JUKAI1, 9, 7, D_RIGHT));
            
            defineWarp("DepartmentD-Escalator/DepartmentA-UpEscalator", new MapLocation(MAP_MALLD, 8, 1), new Warp(MAP_MALLA, 15, 9, 3, true));
            defineWarp("DepartmentD/DepartmentRoof", new MapLocation(MAP_MALLD, 2, 1), new Warp(MAP_MALLROOF, 25, 21, D_RIGHT));
            
            defineWarp("DepartmentRoof/DepartmentD", new MapLocation(MAP_MALLROOF, 1, 1), new Warp(MAP_MALLD, 15, 10, D_LEFT));
            defineWarp("DepartmentRoof/FlyWithBroom", new MapLocation(MAP_MALLROOF, 2, 2), new Warp(MAP_WITCHFLIGHT, 9, 17, D_SAME), false);
            
            defineWarp("FlyWithBroom/DepartmentRoof", new MapLocation(MAP_WITCHFLIGHT, 3, 2), new Warp(MAP_MALLROOF, 40, 22, D_LEFT), false);
            
            defineWarp("EntranceA/EntranceB", new CombinedLocation(new MapLocation(MAP_MALLRECEPTION, 3, 1), new MapLocation(MAP_MALLRECEPTION, 5, 2)),
                                              new Warp(MAP_MALLMUSIC, 14, 9, D_LEFT));
            defineWarp("EntranceA/DepartmentB-LeftDoor", new CombinedLocation(new MapLocation(MAP_MALLRECEPTION, 10, 2), new MapLocation(MAP_MALLRECEPTION, 15, 1)),
                                                         new Warp(MAP_MALLB, 8, 81, D_DOWN));
            
            defineWarp("EntranceB/EntranceA", new CombinedLocation(new MapLocation(MAP_MALLMUSIC, 5, 2), new MapLocation(MAP_MALLMUSIC, 12, 1)),
                                              new Warp(MAP_MALLRECEPTION, 3, 8, D_RIGHT));
            
            defineWarp("EntranceC/DepartmentB-RightDoor", new CombinedLocation(new MapLocation(MAP_MALLTOKUTO, 5, 2), new MapLocation(MAP_MALLTOKUTO, 10, 1)),
                                                          new Warp(MAP_MALLB, 106, 63, D_DOWN));
            
            // 08 Eyeball-Hand
            defineWarp("08EyeballHand/DoorRoom", new MapLocation(MAP_EYEBALLWORLD, 1, 1), new Warp(MAP_NEXUS, 23, 22, D_DOWN));
            defineWarp("08EyeballHand-Mouth/FootprintsA", new MapLocation(MAP_EYEBALLWORLD, 35, 1), new Warp(MAP_FOOTPRINTPATHA, 0, 11, D_RIGHT));
            
            defineWarp("FootprintsA/08EyeballHand-Mouth", new MapLocation(MAP_FOOTPRINTPATHA, 1, 1), new Warp(MAP_EYEBALLWORLD, 83, 54, D_DOWN));
            defineWarp("FootprintsA/FootprintsB", new MapLocation(MAP_FOOTPRINTPATHA, 2, 1), new Warp(MAP_FOOTPRINTPATHB, 0, 19, D_RIGHT));
            
            defineWarp("FootprintsB/FootprintsA", new MapLocation(MAP_FOOTPRINTPATHB, 4, 1), new Warp(MAP_FOOTPRINTPATHA, 49, 11, D_LEFT));
            defineWarp("FootprintsB-RedBoxes/RedA-Footprints", new CombinedLocation(new MapLocation(MAP_FOOTPRINTPATHB, 5, 1), new MapLocation(MAP_FOOTPRINTPATHB, 5, 4)),
                                                               new Warp(MAP_HELL, 36, 35, D_DOWN));
            
            // 09 Small
            defineWarp("09Small/DoorRoom", new MapLocation(MAP_CANDLEWORLD, 1, 1), new Warp(MAP_NEXUS, 25, 18, D_DOWN));
            defineWarp("09Small-Pyramid/CheckerA-Pyramid", new MapLocation(MAP_CANDLEWORLD, 30, 1), new Warp(MAP_CHECKERTILESA, 36, 51, D_DOWN));
            defineWarp("09Small-Toriningen/RedA-Stuck", new MapLocation(MAP_CANDLEWORLD, 9, 11), new Warp(MAP_HELL, 23, 75, 2, "Stuck"), false);
            
            defineWarp("CheckerA-Pyramid/09Small-Pyramid", new MapLocation(MAP_CHECKERTILESA, 4, 1), new Warp(MAP_CANDLEWORLD, 92, 49, D_DOWN));
            defineWarp("CheckerA/CheckerB", new MapLocation(MAP_CHECKERTILESA, 3, 1), new Warp(MAP_CHECKERTILESB, 38, 36, D_LEFT));
            defineWarp("CheckerA-Isolated/LampSmallRoom3-SmallLamp", new MapLocation(MAP_CHECKERTILESA, 7, 1, "Isolated"), new Warp(MAP_LAMPWORLD, 73, 24, D_DOWN));
            defineWarp("CheckerA-Toriningen/LakeCorridor-Stuck", new MapLocation(MAP_CHECKERTILESA, 5, 6), new Warp(MAP_DOCKS, 150, 101, 2, "Stuck"), false);
            
            defineWarp("CheckerB/CheckerA", new MapLocation(MAP_CHECKERTILESB, 5, 1), new Warp(MAP_CHECKERTILESA, 1, 5, D_RIGHT));
            defineWarp("CheckerB-RedBoxes/RedA-Checker", new CombinedLocation(new MapLocation(MAP_CHECKERTILESB, 20, 1), new MapLocation(MAP_CHECKERTILESB, 20, 4)),
                                                         new Warp(MAP_HELL, 21, 10, D_DOWN));
            
            if (derandomizeEvents)
            {
                switch (fixedDreamBedID)
                {
                    case 1: defineWarp("StairsInBed-Upstairs/DreamRoom", new MapLocation(MAP_STAIRCASEOFHANDS, 11, 2), new Warp(MAP_DREAMROOM, 13, 7, D_LEFT)); break;
                    case 2: defineWarp("StairsInBed-Upstairs/02HatAndScarf", new MapLocation(MAP_STAIRCASEOFHANDS, 11, 2, 2), new Warp(MAP_BLOCKWORLD, 42, 58, D_LEFT)); break;
                    case 3: defineWarp("StairsInBed-Upstairs/05YukiOnna", new MapLocation(MAP_STAIRCASEOFHANDS, 11, 2, 3), new Warp(MAP_SNOWWORLD, 91, 49, D_LEFT)); break;
                    case 4: defineWarp("StairsInBed-Upstairs/09Small", new MapLocation(MAP_STAIRCASEOFHANDS, 11, 2, 4), new Warp(MAP_CANDLEWORLD, 68, 87, D_LEFT)); break;
                    case 5: defineWarp("StairsInBed-Upstairs/LampSmallRoom2", new MapLocation(MAP_STAIRCASEOFHANDS, 11, 2, 5), new Warp(MAP_ROOMOFBEDS, 31, 40, D_LEFT)); break;
                }
            }
            // If not derandomized, while it is technically possible to exit to any of the beds, that's only if you can come in from the other side.
            // This is usually not the case; the majority of the time, you can only return to where you were. So just leave those warps out.
            defineWarp("StairsInBed-Downstairs/PassageOutsideStorage", new MapLocation(MAP_STAIRCASEOFHANDS, 15, 1), new Warp(MAP_BEFOREBLAZINGCORRIDOR, 25, 12, D_LEFT));
            
            defineWarp("PassageOutsideStorage/StairsInBed-Downstairs", new MapLocation(MAP_BEFOREBLAZINGCORRIDOR, 4, 1), new Warp(MAP_STAIRCASEOFHANDS, 21, 41, D_RIGHT));
            defineWarp("PassageOutsideStorage-Elevator/DepartmentA-Elevator", new MapLocation(MAP_BEFOREBLAZINGCORRIDOR, 2, 4),
                                                                              new Warp(MAP_MALLA, 4, 6, 2, "One-Way Elevator")); // Not really section, just extra info
            defineWarp("PassageOutsideStorage-Left/OutsideStorage(Flames)", new MapLocation(MAP_BEFOREBLAZINGCORRIDOR, 5, 1), new Warp(MAP_BLAZINGCORRIDOR, 18, 12, D_LEFT));
            
            defineWarp("OutsideStorage(Flames)/PassageOutsideStorage-Left", new MapLocation(MAP_BLAZINGCORRIDOR, 2, 1), new Warp(MAP_BEFOREBLAZINGCORRIDOR, 1, 12, D_RIGHT));
            defineWarp("OutsideStorage(Flames)/Storeroom", new MapLocation(MAP_BLAZINGCORRIDOR, 5, 1), new Warp(MAP_STORAGEROOM, 31, 14, D_LEFT));
            
            defineWarp("Storeroom/OutsideStorage(Flames)", new MapLocation(MAP_STORAGEROOM, 3, 1), new Warp(MAP_BLAZINGCORRIDOR, 6, 12, D_RIGHT));
            defineWarp("Storeroom/SpaceshipEntrance", new MapLocation(MAP_STORAGEROOM, 28, 1), new Warp(MAP_SPACESHIPENTRANCE, 4, 16, D_RIGHT));
            
            defineWarp("SpaceshipEntrance/Storeroom", new MapLocation(MAP_SPACESHIPENTRANCE, 1, 1), new Warp(MAP_STORAGEROOM, 25, 8, D_LEFT));
            defineWarp("SpaceshipEntrance/Spaceship_01", new MapLocation(MAP_SPACESHIPENTRANCE, 3, 1), new Warp(MAP_SPACESHIP01, 2, 11, D_RIGHT), false);
            
            defineWarp("Spaceship_01-LeftFlying/Spaceship_02", new MapLocation(MAP_SPACESHIP01, 1, 1, 2), new Warp(MAP_SPACESHIP02, 15, 15, D_LEFT), false);
            defineWarp("Spaceship_01-LeftLanded/SpaceshipExit", new MapLocation(MAP_SPACESHIP01, 1, 1), new Warp(MAP_SPACESHIPEXIT, 15, 15, D_LEFT), false);
            defineWarp("Spaceship_01-Right/Spaceship_03", new MapLocation(MAP_SPACESHIP01, 3, 1), new Warp(MAP_SPACESHIP03, 4, 16, D_RIGHT), false);
            
            defineWarp("Spaceship_02/Spaceship_01-LeftFlying", new MapLocation(MAP_SPACESHIP02, 3, 1), new Warp(MAP_SPACESHIP01, 2, 11, D_RIGHT), false);
            
            defineWarp("Spaceship_03/Spaceship_01-Right", new MapLocation(MAP_SPACESHIP03, 1, 1), new Warp(MAP_SPACESHIP01, 28, 10, D_LEFT), false);
            
            defineWarp("SpaceshipExit/Spaceship_01-LeftLanded", new MapLocation(MAP_SPACESHIPEXIT, 3, 1), new Warp(MAP_SPACESHIP01, 2, 11, D_RIGHT), false);
            defineWarp("SpaceshipExit/Mars_01", new MapLocation(MAP_SPACESHIPEXIT, 6, 1), new Warp(MAP_MARS01, 38, 9, D_LEFT));
            
            // 10 Cat Ears
            defineWarp("10CatEars/DoorRoom", new MapLocation(MAP_SHIELDFOLKWORLD, 1, 1), new Warp(MAP_NEXUS, 25, 14, D_DOWN));
            defineWarp("10CatEars-Gate/FCEntranceA-Gate", new MapLocation(MAP_SHIELDFOLKWORLD, 3, 1), new Warp(MAP_FCENTRANCEA, 2, 12, D_DOWN));
            
            defineWarp("FCEntranceA-Gate/10CatEars-Gate", new MapLocation(MAP_FCENTRANCEA, 3, 1), new Warp(MAP_SHIELDFOLKWORLD, 60, 53, D_DOWN));
            defineWarp("FCEntranceA-Ladder/FCHouse-Ladder", new MapLocation(MAP_FCENTRANCEA, 2, 1), new Warp(MAP_FCHOUSEOUTSIDE, 9, 11, D_DOWN));
            
            // 11 Neon
            defineWarp("11Neon/DoorRoom", new MapLocation(MAP_NEONWORLD, 1, 1), new Warp(MAP_NEXUS, 23, 10, D_DOWN));
            defineWarp("11Neon-Door/NeonTilesA-Pyramid", new MapLocation(MAP_NEONWORLD, 4, 1), new Warp(MAP_NEONTILESA, 24, 30, D_DOWN));
            
            defineWarp("NeonTilesA-Pyramid/11Neon-Door", new MapLocation(MAP_NEONTILESA, 3, 1), new Warp(MAP_NEONWORLD, 81, 42, D_DOWN));
            defineWarp("NeonTilesA/NeonTilesB", new MapLocation(MAP_NEONTILESA, 4, 1), new Warp(MAP_NEONTILESB, 39, 30, D_LEFT));
            
            defineWarp("NeonTilesB/NeonTilesA", new MapLocation(MAP_NEONTILESB, 1, 1), new Warp(MAP_NEONTILESA, 0, 6, D_RIGHT));
            defineWarp("NeonTilesB-RedBoxes/RedA-NeonTiles", new CombinedLocation(new MapLocation(MAP_NEONTILESB, 2, 1), new MapLocation(MAP_NEONTILESB, 2, 4)),
                                                             new Warp(MAP_HELL, 24, 65, D_DOWN));
            
            // 12 Lamp
            defineWarp("12Lamp/DoorRoom", new MapLocation(MAP_NUMBERWORLD, 22, 1), new Warp(MAP_NEXUS, 19, 8, D_DOWN));
            defineWarp("12Lamp-GuardedDoor/LampSmallRoom1", new MapLocation(MAP_NUMBERWORLD, 36, 1), new Warp(MAP_STABBINGROOM, 21, 9, D_DOWN)); // Requires Knife/Cat to get past guard
            defineWarp("12Lamp-LeftDoor/LampSmallRoom2-Door", new MapLocation(MAP_NUMBERWORLD, 39, 1), new Warp(MAP_ROOMOFBEDS, 46, 10, D_DOWN));
            defineWarp("12Lamp-CanalDoor/LampSmallRoom3-WallDoor", new MapLocation(MAP_NUMBERWORLD, 38, 1, "Canal Door"), new Warp(MAP_LAMPWORLD, 56, 93, D_DOWN));
            defineWarp("12Lamp-Horseshoe/LampSmallRoom3-LonelyDoor", new MapLocation(MAP_NUMBERWORLD, 42, 1), new Warp(MAP_LAMPWORLD, 11, 10, D_DOWN));
            defineWarp("12Lamp-Zipper/LampSmallRoom4", new MapLocation(MAP_NUMBERWORLD, 41, 3), new Warp(MAP_FACESTAIRWAYPASSAGE, 3, 17, D_DOWN)); // Requires Knife to open
            
            defineWarp("LampSmallRoom1/12Lamp-GuardedDoor", new MapLocation(MAP_STABBINGROOM, 186, 1), new Warp(MAP_NUMBERWORLD, 51, 13, D_DOWN));
            
            defineWarp("LampSmallRoom2-Door/12Lamp-LeftDoor", new MapLocation(MAP_ROOMOFBEDS, 1, 1), new Warp(MAP_NUMBERWORLD, 17, 49, D_DOWN));
            if (derandomizeEvents)
            {
                int[] xPos = { 2, 58, 30, 30 };
                int[] yPos = { 26, 26, 49, 3 };
                defineWarp("LampSmallRoom2-ThirdCloset/RedB", new MapLocation(MAP_ROOMOFBEDS, 19, 2),
                    new Warp(MAP_MINIHELL, xPos[fixedMiniHellClosetID - 1], yPos[fixedMiniHellClosetID - 1], D_DOWN)); // Requires Knife to stab Toriningen
            }
            else
            {
                defineWarp("LampSmallRoom2-ThirdCloset/RedB", new MapLocation(MAP_ROOMOFBEDS, 19, 2), new Warp(MAP_MINIHELL, 2, 26, D_DOWN)); // Requires Knife to stab Toriningen
                defineWarp("LampSmallRoom2-ThirdCloset/RedB-2", new MapLocation(MAP_ROOMOFBEDS, 19, 2, 2), new Warp(MAP_MINIHELL, 58, 26, D_DOWN)); // Requires Knife to stab Toriningen
                defineWarp("LampSmallRoom2-ThirdCloset/RedB-3", new MapLocation(MAP_ROOMOFBEDS, 19, 2, 3), new Warp(MAP_MINIHELL, 30, 49, D_DOWN)); // Requires Knife to stab Toriningen
                defineWarp("LampSmallRoom2-ThirdCloset/RedB-4", new MapLocation(MAP_ROOMOFBEDS, 19, 2, 4), new Warp(MAP_MINIHELL, 30, 3, D_DOWN)); // Requires Knife to stab Toriningen
            }
            if (!derandomizeEvents || fixedPurpleVersion != 0)
                defineWarp("LampSmallRoom2-BloodySpotA/PurpleA", new MapLocation(MAP_ROOMOFBEDS, 21, 1), new Warp(MAP_GUILLOTINEWORLDSMALL, 14, 13, D_DOWN));
            if (!derandomizeEvents || fixedPurpleVersion == 0)
                defineWarp("LampSmallRoom2-BloodySpotB/PurpleB", new MapLocation(MAP_ROOMOFBEDS, 21, 1, 2), new Warp(MAP_GUILLOTINEWORLDBIG, 14, 13, D_DOWN));
            defineWarp("LampSmallRoom2-ToriningenA/LampSmallRoom3-AAAA", new MapLocation(MAP_ROOMOFBEDS, 22, 11), new Warp(MAP_LAMPWORLD, 57, 83, 2, "Stuck"), false);
            defineWarp("LampSmallRoom2-ToriningenB/12Lamp-AAAA", new MapLocation(MAP_ROOMOFBEDS, 22, 11, 2), new Warp(MAP_NUMBERWORLD, 33, 16, 2, "Stuck"), false);
            
            if (!derandomizeEvents || fixedPurpleVersion != 0)
            {
                if (derandomizeEvents)
                    defineWarp("PurpleA-Closet/LampSmallRoom2", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, fixedClosetWarpID), new Warp(MAP_ROOMOFBEDS, 20 + ((fixedClosetWarpID - 1) * 2), 36, D_DOWN));
                else
                {
                    defineWarp("PurpleA-Closet/LampSmallRoom2", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2), new Warp(MAP_ROOMOFBEDS, 20, 36, D_DOWN));
                    defineWarp("PurpleA-Closet/LampSmallRoom2-2", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, 2), new Warp(MAP_ROOMOFBEDS, 22, 36, D_DOWN));
                    defineWarp("PurpleA-Closet/LampSmallRoom2-3", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, 3), new Warp(MAP_ROOMOFBEDS, 24, 36, D_DOWN));
                    defineWarp("PurpleA-Closet/LampSmallRoom2-4", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, 4), new Warp(MAP_ROOMOFBEDS, 26, 36, D_DOWN));
                    defineWarp("PurpleA-Closet/LampSmallRoom2-5", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, 5), new Warp(MAP_ROOMOFBEDS, 28, 36, D_DOWN));
                    defineWarp("PurpleA-Closet/LampSmallRoom2-6", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, 6), new Warp(MAP_ROOMOFBEDS, 30, 36, D_DOWN));
                    defineWarp("PurpleA-Closet/LampSmallRoom2-7", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, 7), new Warp(MAP_ROOMOFBEDS, 32, 36, D_DOWN));
                    defineWarp("PurpleA-Closet/LampSmallRoom2-8", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 1, 2, 8), new Warp(MAP_ROOMOFBEDS, 34, 36, D_DOWN));
                }
                defineWarp("PurpleA-PantsToriningen/RedA-Stuck", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 8, 6), new Warp(MAP_HELL, 38, 38, 2, "Stuck"), false);
                defineWarp("PurpleA-DressToriningen/Highway-Stuck", new MapLocation(MAP_GUILLOTINEWORLDSMALL, 5, 6), new Warp(MAP_HIGHWAY, 153, 20, 2, "Stuck"), false);
            }
            
            if (!derandomizeEvents || fixedPurpleVersion == 0)
            {
                if (derandomizeEvents)
                    defineWarp("PurpleB-Closet/LampSmallRoom2", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, fixedClosetWarpID), new Warp(MAP_ROOMOFBEDS, 20 + ((fixedClosetWarpID - 1) * 2), 36, D_DOWN));
                else
                {
                    defineWarp("PurpleB-Closet/LampSmallRoom2", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2), new Warp(MAP_ROOMOFBEDS, 20, 36, D_DOWN));
                    defineWarp("PurpleB-Closet/LampSmallRoom2-2", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, 2), new Warp(MAP_ROOMOFBEDS, 22, 36, D_DOWN));
                    defineWarp("PurpleB-Closet/LampSmallRoom2-3", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, 3), new Warp(MAP_ROOMOFBEDS, 24, 36, D_DOWN));
                    defineWarp("PurpleB-Closet/LampSmallRoom2-4", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, 4), new Warp(MAP_ROOMOFBEDS, 26, 36, D_DOWN));
                    defineWarp("PurpleB-Closet/LampSmallRoom2-5", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, 5), new Warp(MAP_ROOMOFBEDS, 28, 36, D_DOWN));
                    defineWarp("PurpleB-Closet/LampSmallRoom2-6", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, 6), new Warp(MAP_ROOMOFBEDS, 30, 36, D_DOWN));
                    defineWarp("PurpleB-Closet/LampSmallRoom2-7", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, 7), new Warp(MAP_ROOMOFBEDS, 32, 36, D_DOWN));
                    defineWarp("PurpleB-Closet/LampSmallRoom2-8", new MapLocation(MAP_GUILLOTINEWORLDBIG, 1, 2, 8), new Warp(MAP_ROOMOFBEDS, 34, 36, D_DOWN));
                }
                defineWarp("PurpleB-PantsToriningen/RedA-Stuck", new CombinedLocation(new MapLocation(MAP_GUILLOTINEWORLDBIG, 3, 6),
                                                                                      new MapLocation(MAP_GUILLOTINEWORLDBIG, 21, 6),
                                                                                      new MapLocation(MAP_GUILLOTINEWORLDBIG, 22, 6),
                                                                                      new MapLocation(MAP_GUILLOTINEWORLDBIG, 23, 6)),
                                                                 new Warp(MAP_HELL, 38, 38, 2, "Stuck"), false);
                defineWarp("PurpleB-DressToriningen/Highway-Stuck", new CombinedLocation(new MapLocation(MAP_GUILLOTINEWORLDBIG, 5, 6),
                                                                                         new MapLocation(MAP_GUILLOTINEWORLDBIG, 24, 6),
                                                                                         new MapLocation(MAP_GUILLOTINEWORLDBIG, 25, 6),
                                                                                         new MapLocation(MAP_GUILLOTINEWORLDBIG, 26, 6)),
                                                                    new Warp(MAP_HIGHWAY, 153, 20, 2, "Stuck"), false);
            }
            
            defineWarp("LampSmallRoom3-WallDoor/12Lamp-CanalDoor", new MapLocation(MAP_LAMPWORLD, 1, 1), new Warp(MAP_NUMBERWORLD, 117, 86, 2, "Canal Door"));
            defineWarp("LampSmallRoom3-LonelyDoor/12Lamp-Horseshoe", new MapLocation(MAP_LAMPWORLD, 8, 1), new Warp(MAP_NUMBERWORLD, 108, 56, D_DOWN));
            defineWarp("LampSmallRoom3-SmallLamp/CheckerA-Isolated", new MapLocation(MAP_LAMPWORLD, 2, 1), new Warp(MAP_CHECKERTILESA, 25, 29, 2, "Isolated"));
            
            defineWarp("LampSmallRoom4/12Lamp-Zipper", new MapLocation(MAP_FACESTAIRWAYPASSAGE, 2, 1), new Warp(MAP_NUMBERWORLD, 133, 63, D_DOWN));
            defineWarp("LampSmallRoom4/LampSmallRoom5", new MapLocation(MAP_FACESTAIRWAYPASSAGE, 6, 1), new Warp(MAP_FACESTAIRWAY, 0, 14, D_RIGHT));
            
            defineWarp("LampSmallRoom5/LampSmallRoom4", new MapLocation(MAP_FACESTAIRWAY, 2, 1), new Warp(MAP_FACESTAIRWAYPASSAGE, 49, 15, D_LEFT));
            defineWarp("LampSmallRoom5/Image", new MapLocation(MAP_FACESTAIRWAY, 3, 1), new Warp(MAP_FACE, 8, 6, D_DOWN), false);
            
            // Sub-Maps
            
            // Red
            defineWarp("RedA-NeonTiles/NeonTilesB-RedBoxes", new MapLocation(MAP_HELL, 3, 1), new Warp(MAP_NEONTILESB, 13, 20, D_DOWN));
            defineWarp("RedA-Thorns/Thorns-RedBoxes", new MapLocation(MAP_HELL, 4, 1), new Warp(MAP_FACECARPETPLAZA, 31, 25, D_DOWN));
            defineWarp("RedA-Footprints/FootprintsB-RedBoxes", new MapLocation(MAP_HELL, 6, 1), new Warp(MAP_FOOTPRINTPATHB, 26, 42, D_DOWN));
            defineWarp("RedA-JukaiGate/Jukai2-Gate", new MapLocation(MAP_HELL, 9, 1), new Warp(MAP_JUKAI2, 38, 12, 2, "Top-Right"));
            defineWarp("RedA-LakeStairs/LakeCorridor-RightSide", new MapLocation(MAP_HELL, 10, 1), new Warp(MAP_DOCKS, 96, 76, 1, "East Side"));
            defineWarp("RedA-Checker/CheckerB-RedBoxes", new MapLocation(MAP_HELL, 14, 1), new Warp(MAP_CHECKERTILESB, 18, 7, D_DOWN));
            defineWarp("RedA-Henkei/FootprintsA-Henkei", new CombinedLocation(new MapLocation(MAP_HELL, 11, 1), new MapLocation(MAP_HELL, 12, 1), new MapLocation(MAP_HELL, 13, 1)),
                                                         new Warp(MAP_FOOTPRINTPATHA, 8, 11, D_UP)); // Requires Knife to stab
            defineWarp("RedA-PantsToriningen/Siblings-Stuck", new MapLocation(MAP_HELL, 2, 6), new Warp(MAP_WARPMAZE, 80, 37, 2, "Stuck"), false);
            defineWarp("RedA-SkirtToriningen/Highway-Stuck", new MapLocation(MAP_HELL, 1, 6), new Warp(MAP_HIGHWAY, 153, 20, 2, "Stuck"), false);
            defineWarp("RedA-DressToriningen/Park-Stuck", new MapLocation(MAP_HELL, 8, 6), new Warp(MAP_PARK, 7, 4, 2, "Stuck"), false);
            
            if (derandomizeEvents)
                defineWarp("RedB-Closet/LampSmallRoom2", new MapLocation(MAP_MINIHELL, 1, 2, fixedClosetWarpID), new Warp(MAP_ROOMOFBEDS, 20 + ((fixedClosetWarpID - 1) * 2), 36, D_DOWN));
            else
            {
                defineWarp("RedB-Closet/LampSmallRoom2", new MapLocation(MAP_MINIHELL, 1, 2), new Warp(MAP_ROOMOFBEDS, 20, 36, D_DOWN));
                defineWarp("RedB-Closet/LampSmallRoom2-2", new MapLocation(MAP_MINIHELL, 1, 2, 2), new Warp(MAP_ROOMOFBEDS, 22, 36, D_DOWN));
                defineWarp("RedB-Closet/LampSmallRoom2-3", new MapLocation(MAP_MINIHELL, 1, 2, 3), new Warp(MAP_ROOMOFBEDS, 24, 36, D_DOWN));
                defineWarp("RedB-Closet/LampSmallRoom2-4", new MapLocation(MAP_MINIHELL, 1, 2, 4), new Warp(MAP_ROOMOFBEDS, 26, 36, D_DOWN));
                defineWarp("RedB-Closet/LampSmallRoom2-5", new MapLocation(MAP_MINIHELL, 1, 2, 5), new Warp(MAP_ROOMOFBEDS, 28, 36, D_DOWN));
                defineWarp("RedB-Closet/LampSmallRoom2-6", new MapLocation(MAP_MINIHELL, 1, 2, 6), new Warp(MAP_ROOMOFBEDS, 30, 36, D_DOWN));
                defineWarp("RedB-Closet/LampSmallRoom2-7", new MapLocation(MAP_MINIHELL, 1, 2, 7), new Warp(MAP_ROOMOFBEDS, 32, 36, D_DOWN));
                defineWarp("RedB-Closet/LampSmallRoom2-8", new MapLocation(MAP_MINIHELL, 1, 2, 8), new Warp(MAP_ROOMOFBEDS, 34, 36, D_DOWN));
            }
            
            // Earth
            defineWarp("Earth-Fence/SandF-Fence", new MapLocation(MAP_BARRACKSSETTLEMENT, 27, 1), new Warp(MAP_WILDERNESSTOSETTLEMENT, 30, 9, D_DOWN));
            defineWarp("Earth-UpperHouse/EarthA-RightDoor", new MapLocation(MAP_BARRACKSSETTLEMENT, 26, 1), new Warp(MAP_BARRACKSSHACKA, 86, 8, D_LEFT));
            defineWarp("Earth-LowerHouse/EarthB", new MapLocation(MAP_BARRACKSSETTLEMENT, 69, 1), new Warp(MAP_BARRACKSSHACKB, 25, 8, D_LEFT));
            defineWarp("Earth-RandomFigure/FCVillage003", new MapLocation(MAP_BARRACKSSETTLEMENT, 70, 1), new Warp(MAP_FCPIRORIVILLAGE, 9, 8, D_DOWN));
            
            defineWarp("EarthA-RightDoor/Earth-UpperHouse", new MapLocation(MAP_BARRACKSSHACKA, 1, 1), new Warp(MAP_BARRACKSSETTLEMENT, 9, 10, D_RIGHT));
            defineWarp("EarthA-LeftDoor/06Hairstyle", new MapLocation(MAP_BARRACKSSHACKA, 8, 1), new Warp(MAP_MURALWORLD, 96, 109, D_DOWN));
            
            defineWarp("EarthB/Earth-LowerHouse", new MapLocation(MAP_BARRACKSSHACKB, 1, 1), new Warp(MAP_BARRACKSSETTLEMENT, 5, 50, D_RIGHT));
            
            // Jukai
            defineWarp("Jukai1/Jukai2-Left", new MapLocation(MAP_JUKAI1, 1, 1), new Warp(MAP_JUKAI2, 0, 12, 1, "Top-Left"));
            defineWarp("Jukai1-Hole/DepartmentC-Hole", new CombinedLocation(new MapLocation(MAP_JUKAI1, 2, 1), new MapLocation(MAP_JUKAI1, 3, 1)),
                                                       new Warp(MAP_MALLC, 6, 10, D_RIGHT));
            
            defineWarp("Jukai2-Left/Jukai1", new MapLocation(MAP_JUKAI2, 3, 1, "Top-Left"), new Warp(MAP_JUKAI1, 19, 9, D_LEFT));
            defineWarp("Jukai2-Jellyfish/Siblings", new MapLocation(MAP_JUKAI2, 14, 1), new Warp(MAP_WARPMAZE, 17, 41, 2, "Top-Left"));
            defineWarp("Jukai2-Right/Jukai3-Top", new MapLocation(MAP_JUKAI2, 13, 1, "Top-Right"), new Warp(MAP_JUKAI3, 0, 9, D_DOWN));
            defineWarp("Jukai2-Gate/RedA-JukaiGate", new MapLocation(MAP_JUKAI2, 12, 1, "Top-Right"), new Warp(MAP_HELL, 147, 24, D_DOWN));
            defineWarp("Jukai2-RoadRight/HighwayD", new CombinedLocation(new MapLocation(MAP_JUKAI2, 4, 1, "Road"), new MapLocation(MAP_JUKAI2, 25, 1, "Road")),
                                                    new Warp(MAP_HIGHWAYD, 0, 15, D_RIGHT), false); // Merged Y=15/Y=16 warps
            defineWarp("Jukai2-WormHole/Mono010-LeftWorm", new CombinedLocation(new MapLocation(MAP_JUKAI2, 26, 1, "Road"),
                                                                                new MapLocation(MAP_JUKAI2, 26, 2, "Road"),
                                                                                new MapLocation(MAP_JUKAI2, 27, 1, "Road"),
                                                                                new MapLocation(MAP_JUKAI2, 27, 2, "Road")),
                                                           new Warp(MAP_MONO010, 2, 12, D_DOWN));
            // Note: Manhole is only open if you walked long enough on Infinite Road (5 variable increases).
            // Thus, you can't access it without access to the Infinite Road, and that's why HighwayD is not shuffled.
            
            defineWarp("Siblings/SandG-BehindFence", new MapLocation(MAP_WARPMAZE, 62, 1), new Warp(MAP_WILDERNESSFCPORTAL, 15, 18, 2, "Behind Fence"));
            // Also has many warps within same map.
            
            defineWarp("Jukai3-Top/Jukai2-Right", new MapLocation(MAP_JUKAI3, 12, 1), new Warp(MAP_JUKAI2, 79, 11, 3, "Top-Right"));
            defineWarp("Jukai3-Bottom/JukaiTrain", new CombinedLocation(new MapLocation(MAP_JUKAI3, 1, 1), new MapLocation(MAP_JUKAI3, 2, 1)),
                                                   new Warp(MAP_JUKAITRAINBEFORE, 19, 10, D_LEFT));
            
            defineWarp("JukaiTrain/Jukai3-Bottom", new MapLocation(MAP_JUKAITRAINBEFORE, 2, 1), new Warp(MAP_JUKAI3, 0, 44, D_RIGHT));
            defineWarp("JukaiTrain/TrainInterior-PreTrip", new MapLocation(MAP_JUKAITRAINBEFORE, 1, 1), new Warp(MAP_TRAININTERIOR, 5, 10, 0, "Pre-Trip")); // Extra info, not section
            
            defineWarp("TrainInterior-PreTrip/JukaiTrain", new MapLocation(MAP_TRAININTERIOR, 1, 1), new Warp(MAP_JUKAITRAINBEFORE, 5, 10, D_DOWN));
            defineWarp("TrainInterior-PostTrip/JukaiTrainB", new MapLocation(MAP_TRAININTERIOR, 1, 1, 2), new Warp(MAP_JUKAITRAINAFTER, 5, 10, D_DOWN));
            
            defineWarp("JukaiTrainB/TrainInterior-PostTrip", new MapLocation(MAP_JUKAITRAINAFTER, 1, 1), new Warp(MAP_TRAININTERIOR, 5, 10, 0, "Post-Trip")); // Extra info, not section
            defineWarp("JukaiTrainB/Jukai4", new MapLocation(MAP_JUKAITRAINAFTER, 2, 1), new Warp(MAP_JUKAI4, 39, 3, D_LEFT));
            
            defineWarp("Jukai4/JukaiTrainB", new CombinedLocation(new MapLocation(MAP_JUKAI4, 7, 1), new MapLocation(MAP_JUKAI4, 8, 1)),
                                             new Warp(MAP_JUKAITRAINAFTER, 0, 10, D_RIGHT));
            defineWarp("Jukai4/Bridge", new CombinedLocation(new MapLocation(MAP_JUKAI4, 5, 1), new MapLocation(MAP_JUKAI4, 6, 1)),
                                        new Warp(MAP_WITCHBRIDGE, 119, 10, D_LEFT)); // Merged Y=9/Y=10 warps
            
            defineWarp("Bridge/Jukai4", new CombinedLocation(new MapLocation(MAP_WITCHBRIDGE, 1, 1), new MapLocation(MAP_WITCHBRIDGE, 3, 1)),
                                        new Warp(MAP_JUKAI4, 12, 21, D_RIGHT)); // Merged Y=21/Y=22 warps
            defineWarp("Bridge/Shore", new CombinedLocation(new MapLocation(MAP_WITCHBRIDGE, 2, 1), new MapLocation(MAP_WITCHBRIDGE, 4, 1)),
                                       new Warp(MAP_WITCHISLAND, 39, 8, D_LEFT)); // Merged Y=8/Y=9 warps
            
            defineWarp("Shore/Bridge", new CombinedLocation(new MapLocation(MAP_WITCHISLAND, 1, 1), new MapLocation(MAP_WITCHISLAND, 2, 1)),
                                       new Warp(MAP_WITCHBRIDGE, 0, 10, D_RIGHT)); // Merged Y=9/Y=10 warps
            
            // Mono
            defineWarp("Mono-Mouth/02HatAndScarf-Gate", new MapLocation(MAP_MONO, 7, 1), new Warp(MAP_BLOCKWORLD, 89, 71, D_DOWN));
            defineWarp("Mono-TunnelLeft/Mono001-Left", new MapLocation(MAP_MONO, 4, 1), new Warp(MAP_MONO001, 0, 8, 1, "Left"));
            defineWarp("Mono-TunnelRight/Mono001-Right", new MapLocation(MAP_MONO, 3, 1), new Warp(MAP_MONO001, 19, 8, 3, "Right"));
            defineWarp("Mono-LoopTunnelLeft/Mono003-TunnelRight", new MapLocation(MAP_MONO, 12, 1), new Warp(MAP_MONO003, 30, 32, D_RIGHT)); // Requires stabbing Dave Spector
            defineWarp("Mono-LoopTunnelRight/Mono002-Right", new MapLocation(MAP_MONO, 12, 1, 2), new Warp(MAP_MONO002, 49, 11, D_LEFT)); // Requires stabbing Dave Spector
            defineWarp("Mono-EyeBox/Mono005", new MapLocation(MAP_MONO, 8, 1), new Warp(MAP_MONO005, 10, 23, D_UP));
            
            defineWarp("Mono001-Left/Mono-TunnelLeft", new MapLocation(MAP_MONO001, 3, 1, "Left"), new Warp(MAP_MONO, 46, 17, D_LEFT));
            defineWarp("Mono001-Right/Mono-TunnelRight", new MapLocation(MAP_MONO001, 4, 1, "Right"), new Warp(MAP_MONO, 51, 17, D_RIGHT));
            
            defineWarp("Mono002-Right/Mono-LoopTunnelRight", new MapLocation(MAP_MONO002, 4, 1), new Warp(MAP_MONO, 17, 64, D_RIGHT));
            defineWarp("Mono002-Left/Mono006-Right", new MapLocation(MAP_MONO002, 3, 1), new Warp(MAP_MONO006, 30, 13, D_LEFT));
            
            defineWarp("Mono003-TunnelLeft/Mono-LoopTunnelRight", new MapLocation(MAP_MONO003, 3, 1), new Warp(MAP_MONO, 17, 64, D_RIGHT));
            defineWarp("Mono003-TunnelRight/Mono-LoopTunnelLeft", new MapLocation(MAP_MONO003, 3, 1, 2), new Warp(MAP_MONO, 15, 64, D_LEFT));
            
            defineWarp("Mono004/Mono006-Left", new MapLocation(MAP_MONO004, 1, 1), new Warp(MAP_MONO006, 0, 13, D_RIGHT));
            
            defineWarp("Mono005/Mono-EyeBox", new MapLocation(MAP_MONO005, 27, 1), new Warp(MAP_MONO, 44, 43, D_UP));
            
            defineWarp("Mono006-Right/Mono002-Left", new MapLocation(MAP_MONO006, 2, 1), new Warp(MAP_MONO002, 0, 11, D_RIGHT));
            defineWarp("Mono006-Left/Mono004", new MapLocation(MAP_MONO006, 1, 1), new Warp(MAP_MONO004, 19, 7, D_LEFT));
            
            defineWarp("Mono008-Left/Mono011-Head", new MapLocation(MAP_MONO008, 1, 1), new Warp(MAP_MONO011, 39, 40, 3, "Inside Circle"));
            defineWarp("Mono008/Mono009", new MapLocation(MAP_MONO008, 26, 1), new Warp(MAP_MONO009, 3, 13, D_RIGHT));
            
            defineWarp("Mono009/Mono008", new MapLocation(MAP_MONO009, 1, 1), new Warp(MAP_MONO008, 21, 10, D_LEFT));
            
            defineWarp("Mono010-LeftWorm/Jukai2-WormHole", new MapLocation(MAP_MONO010, 3, 1), new Warp(MAP_JUKAI2, 61, 28, 2, "Road"));
            defineWarp("Mono010-RightWorm/Mono011-WormHole", new MapLocation(MAP_MONO010, 1, 1), new Warp(MAP_MONO011, 31, 40, 1, "Inside Circle"));
            
            defineWarp("Mono011-WormHole/Mono010-RightWorm", new CombinedLocation(new MapLocation(MAP_MONO011, 1, 1, "Inside Circle"), new MapLocation(MAP_MONO011, 3, 1, "Inside Circle")),
                                                             new Warp(MAP_MONO010, 16, 12, D_DOWN));
            defineWarp("Mono011-Head/Mono008-Left", new MapLocation(MAP_MONO011, 4, 1, "Inside Circle"), new Warp(MAP_MONO008, 0, 12, D_RIGHT));
            defineWarp("Mono011-Tunnel/Mono-HandPath", new MapLocation(MAP_MONO011, 7, 1, "Outside Circle"), new Warp(MAP_MONO, 93, 94, D_RIGHT));
            defineWarp("Mono011-Stairs/Mono013-Stairs", new MapLocation(MAP_MONO011, 19, 2, "Outside Circle"), new Warp(MAP_MONO013, 18, 5, D_LEFT));
            
            defineWarp("Mono012/Mono013", new MapLocation(MAP_MONO012, 1, 1), new Warp(MAP_MONO013, 0, 11, D_RIGHT));
            defineWarp("Mono012-Stairs/FCDungeon003", new MapLocation(MAP_MONO012, 8, 1), new Warp(MAP_FCDUNGEON003, 4, 8, D_DOWN));
            
            defineWarp("Mono013-Stairs/Mono011-Stairs", new MapLocation(MAP_MONO013, 16, 1), new Warp(MAP_MONO011, 17, 69, 1, "Outside Circle"));
            defineWarp("Mono013/Mono012", new MapLocation(MAP_MONO013, 8, 1), new Warp(MAP_MONO012, 35, 16, D_LEFT));
            
            // Shallows
            defineWarp("Shallows/Igloo7-Pond", new MapLocation(MAP_PINKSEA, 7, 1), new Warp(MAP_IGLOO7, 8, 8, D_LEFT));
            defineWarp("Shallows/ShallowsHouse", new MapLocation(MAP_PINKSEA, 12, 1), new Warp(MAP_PONIKOHOUSE, 3, 10, D_RIGHT));
            // Also has many warps within same map.
            
            defineWarp("ShallowsHouse/Shallows", new MapLocation(MAP_PONIKOHOUSE, 1, 1), new Warp(MAP_PINKSEA, 59, 100, D_DOWN));
            defineWarp("ShallowsHouse/Mono007", new MapLocation(MAP_PONIKOHOUSE, 3, 5), new Warp(MAP_MONO007, 9, 13, D_LEFT), false);
            
            // Lake Corridor
            defineWarp("LakeCorridor-ByVending/Gutter_005-Stairs", new MapLocation(MAP_DOCKS, 83, 1, "West Side"), new Warp(MAP_GUTTER005, 60, 14, D_LEFT));
            defineWarp("LakeCorridor-RightSide/RedA-LakeStairs", new MapLocation(MAP_DOCKS, 87, 1, "East Side"), new Warp(MAP_HELL, 163, 58, D_RIGHT));
            defineWarp("LakeCorridor-Figure/SandF", new MapLocation(MAP_DOCKS, 67, 1, "East Side"), new Warp(MAP_WILDERNESSTOSETTLEMENT, 10, 45, D_DOWN));
            
            // Park
            defineWarp("Park-Stairs/SandStairs", new MapLocation(MAP_PARK, 8, 1), new Warp(MAP_STAIRWAYTOTHESKY, 198, 12, D_LEFT));
            defineWarp("Park/ParkCliff", new CombinedLocation(new MapLocation(MAP_PARK, 23, 1), new MapLocation(MAP_PARK, 24, 1), new MapLocation(MAP_PARK, 40, 1)),
                                         new Warp(MAP_PARKCLIFF, 0, 9, D_RIGHT)); // Merged Y=8/Y=9/Y=10 warps
            
            defineWarp("ParkCliff/Park", new CombinedLocation(new MapLocation(MAP_PARKCLIFF, 4, 1), new MapLocation(MAP_PARKCLIFF, 5, 1), new MapLocation(MAP_PARKCLIFF, 6, 1)),
                                         new Warp(MAP_PARK, 69, 24, D_LEFT)); // Merged Y=23/Y=24/Y=25 warps
            defineWarp("ParkCliff/InsideBricksA", new MapLocation(MAP_PARKCLIFF, 14, 1), new Warp(MAP_INSIDEBRICKSA, 2, 11, D_RIGHT));
            
            defineWarp("InsideBricksA/ParkCliff", new MapLocation(MAP_INSIDEBRICKSA, 1, 1), new Warp(MAP_PARKCLIFF, 13, 6, D_LEFT));
            defineWarp("InsideBricksA-Darkness/InsideBricksB-Normal", new MapLocation(MAP_INSIDEBRICKSA, 2, 1), new Warp(MAP_INSIDEBRICKSB, 11, 11, D_RIGHT), false); // Instant warp
            
            defineWarp("InsideBricksB-Normal/InsideBricksA-Darkness", new MapLocation(MAP_INSIDEBRICKSB, 5, 1, 2), new Warp(MAP_INSIDEBRICKSA, 26, 11, D_LEFT), false); // Instant warp
            defineWarp("InsideBricksB-Flower/InsideBricksC-Darkness", new MapLocation(MAP_INSIDEBRICKSB, 5, 1), new Warp(MAP_INSIDEBRICKSC, 21, 11, D_LEFT), false); // Instant warp
            
            defineWarp("InsideBricksC-Darkness/InsideBricksB-Flower", new MapLocation(MAP_INSIDEBRICKSC, 3, 1), new Warp(MAP_INSIDEBRICKSB, 16, 11, D_RIGHT), false); // Instant warp
            defineWarp("InsideBricksC/Wall", new MapLocation(MAP_INSIDEBRICKSC, 1, 1), new Warp(MAP_GHOSTWORLD, 16, 7, D_LEFT));
            
            defineWarp("Wall/InsideBricksC", new MapLocation(MAP_GHOSTWORLD, 16, 1), new Warp(MAP_INSIDEBRICKSC, 3, 11, D_RIGHT));
            
            // Mars
            defineWarp("Mars_01/SpaceshipExit", new MapLocation(MAP_MARS01, 2, 1), new Warp(MAP_SPACESHIPEXIT, 11, 16, D_RIGHT));
            defineWarp("Mars_01/Mars_02", new MapLocation(MAP_MARS01, 12, 1), new Warp(MAP_MARS02, 38, 14, D_LEFT));
            
            defineWarp("Mars_02/Mars_01", new MapLocation(MAP_MARS02, 1, 1), new Warp(MAP_MARS01, 1, 14, D_RIGHT));
            defineWarp("Mars_02/Mars_03", new MapLocation(MAP_MARS02, 2, 1), new Warp(MAP_MARS03, 38, 14, D_LEFT));
            
            defineWarp("Mars_03/Mars_02", new MapLocation(MAP_MARS03, 1, 1), new Warp(MAP_MARS02, 1, 14, D_RIGHT));
            defineWarp("Mars_03/Mars_04", new MapLocation(MAP_MARS03, 6, 1), new Warp(MAP_MARS04, 16, 13, D_LEFT));
            
            defineWarp("Mars_04/Mars_03", new MapLocation(MAP_MARS04, 1, 1), new Warp(MAP_MARS03, 1, 13, D_RIGHT));
            defineWarp("Mars_04/Mars_05", new MapLocation(MAP_MARS04, 14, 1), new Warp(MAP_MARS05, 16, 13, D_LEFT));
            
            defineWarp("Mars_05/Mars_04", new MapLocation(MAP_MARS05, 1, 1), new Warp(MAP_MARS04, 4, 1, D_RIGHT));
            defineWarp("Mars_05-Small/MarsBasement_01", new MapLocation(MAP_MARS05, 6, 2), new Warp(MAP_MARSBASEMENTSTAIRS, 29, 1, D_LEFT)); // Requires Small
            
            defineWarp("MarsBasement_01/Mars_05-Small", new MapLocation(MAP_MARSBASEMENTSTAIRS, 23, 1), new Warp(MAP_MARS05, 11, 12, D_RIGHT)); // Requires Small
            defineWarp("MarsBasement_01/MarsBasement_02", new MapLocation(MAP_MARSBASEMENTSTAIRS, 44, 1), new Warp(MAP_MARSBASEMENT, 1, 10, D_RIGHT));
            
            defineWarp("MarsBasement_02/MarsBasement_01", new MapLocation(MAP_MARSBASEMENT, 1, 1), new Warp(MAP_MARSBASEMENTSTAIRS, 27, 23, D_LEFT));
            
            // FC Maps
            defineWarp("FCHouse-Ladder/FCEntranceA-Ladder", new MapLocation(MAP_FCHOUSEOUTSIDE, 1, 1), new Warp(MAP_FCENTRANCEA, 18, 6, D_DOWN));
            defineWarp("FCHouse/FCInsideHouse", new MapLocation(MAP_FCHOUSEOUTSIDE, 2, 1), new Warp(MAP_FCHOUSEINSIDE, 12, 8, D_DOWN));
            
            defineWarp("FCInsideHouse/FCHouse", new MapLocation(MAP_FCHOUSEINSIDE, 1, 1), new Warp(MAP_FCHOUSEOUTSIDE, 28, 8, D_DOWN));
            defineWarp("FCInsideHouse/FCBasement001-Stairs", new MapLocation(MAP_FCHOUSEINSIDE, 4, 1), new Warp(MAP_FCBASEMENT001, 21, 6, D_LEFT));
            
            // Almost all basement maps are left out of shuffle for, well, seed quality reasons.
            defineWarp("FCBasement001-Stairs/FCInsideHouse", new MapLocation(MAP_FCBASEMENT001, 1, 1), new Warp(MAP_FCHOUSEINSIDE, 5, 9, D_RIGHT));
            defineWarp("FCBasement001-RightDoor/FCBasement002-UpstairsDoor", new MapLocation(MAP_FCBASEMENT001, 8, 1), new Warp(MAP_FCBASEMENT002, 9, 4, D_DOWN), false);
            defineWarp("FCBasement001-LeftDoor/FCBasement003-LeftDoor", new MapLocation(MAP_FCBASEMENT001, 9, 1), new Warp(MAP_FCBASEMENT003, 6, 7, D_DOWN), false);
            
            defineWarp("FCBasement002-UpstairsDoor/FCBasement001-RightDoor", new MapLocation(MAP_FCBASEMENT002, 11, 1), new Warp(MAP_FCBASEMENT001, 12, 8, D_DOWN), false);
            defineWarp("FCBasement002-RightDoor/FCBasement003-RightDoor", new MapLocation(MAP_FCBASEMENT002, 12, 1), new Warp(MAP_FCBASEMENT003, 16, 9, D_DOWN), false);
            defineWarp("FCBasement002-LeftDoor/FCBasement006-Door", new MapLocation(MAP_FCBASEMENT002, 13, 1), new Warp(MAP_FCBASEMENT006, 7, 8, D_DOWN), false);
            
            defineWarp("FCBasement003-LeftDoor/FCBasement001-LeftDoor", new MapLocation(MAP_FCBASEMENT003, 1, 1), new Warp(MAP_FCBASEMENT001, 8, 8, D_DOWN), false);
            defineWarp("FCBasement003-RightDoor/FCBasement002-RightDoor", new MapLocation(MAP_FCBASEMENT003, 2, 1), new Warp(MAP_FCBASEMENT002, 17, 10, D_DOWN), false);
            defineWarp("FCBasement003-MiddleDoor/FCBasement004-Right", new MapLocation(MAP_FCBASEMENT003, 3, 1), new Warp(MAP_FCBASEMENT004, 39, 9, D_LEFT), false);
            
            defineWarp("FCBasement004-Right/FCBasement003-MiddleDoor", new MapLocation(MAP_FCBASEMENT004, 14, 1), new Warp(MAP_FCBASEMENT003, 10, 7, D_DOWN), false);
            defineWarp("FCBasement004-Left/FCBasement005-LeftDoor", new MapLocation(MAP_FCBASEMENT004, 30, 1), new Warp(MAP_FCBASEMENT005, 7, 8, D_DOWN), false);
            
            defineWarp("FCBasement005-LeftDoor/FCBasement004-Left", new MapLocation(MAP_FCBASEMENT005, 1, 1), new Warp(MAP_FCBASEMENT004, 0, 7, D_RIGHT), false);
            defineWarp("FCBasement005-RightDoor/FCBasement007-RightDoor", new MapLocation(MAP_FCBASEMENT005, 2, 1), new Warp(MAP_FCBASEMENT007, 12, 8, D_DOWN), false);
            
            defineWarp("FCBasement006-Door/FCBasement002-LeftDoor", new MapLocation(MAP_FCBASEMENT006, 1, 1), new Warp(MAP_FCBASEMENT002, 13, 10, D_DOWN), false);
            defineWarp("FCBasement006-Right/FCBasement007-Left", new MapLocation(MAP_FCBASEMENT006, 2, 1), new Warp(MAP_FCBASEMENT007, 0, 9, D_RIGHT), false);
            defineWarp("FCBasement006-Stairs/FCBasement008-Stairs", new MapLocation(MAP_FCBASEMENT006, 9, 1), new Warp(MAP_FCBASEMENT008, 1, 5, D_RIGHT), false);
            
            defineWarp("FCBasement007-RightDoor/FCBasement005-RightDoor", new MapLocation(MAP_FCBASEMENT007, 1, 1), new Warp(MAP_FCBASEMENT005, 12, 8, D_DOWN), false);
            defineWarp("FCBasement007-Left/FCBasement006-Right", new MapLocation(MAP_FCBASEMENT007, 2, 1), new Warp(MAP_FCBASEMENT006, 19, 8, D_LEFT), false);
            defineWarp("FCBasement007-LeftDoor/FCBasement008-RightDoor", new MapLocation(MAP_FCBASEMENT007, 3, 1), new Warp(MAP_FCBASEMENT008, 12, 8, D_DOWN), false);
            
            defineWarp("FCBasement008-LeftDoor/FCBasement011-Door", new MapLocation(MAP_FCBASEMENT008, 1, 1), new Warp(MAP_FCBASEMENT011, 4, 6, D_DOWN), false);
            defineWarp("FCBasement008-RightDoor/FCBasement007-LeftDoor", new MapLocation(MAP_FCBASEMENT008, 2, 1), new Warp(MAP_FCBASEMENT007, 7, 8, D_DOWN), false);
            defineWarp("FCBasement008-Stairs/FCBasement006-Stairs", new MapLocation(MAP_FCBASEMENT008, 3, 1), new Warp(MAP_FCBASEMENT006, 16, 13, D_LEFT), false);
            defineWarp("FCBasement008-Right/FCBasement013-Left", new MapLocation(MAP_FCBASEMENT008, 7, 1), new Warp(MAP_FCBASEMENT013, 0, 8, D_RIGHT), false);
            
            defineWarp("FCBasement009-LeftStairs/FCBasement013-Stairs", new MapLocation(MAP_FCBASEMENT009, 1, 1), new Warp(MAP_FCBASEMENT013, 18, 12, D_LEFT), false);
            defineWarp("FCBasement009-Right/FCBasement012-Left", new MapLocation(MAP_FCBASEMENT009, 2, 1), new Warp(MAP_FCBASEMENT012, 0, 8, D_RIGHT), false);
            defineWarp("FCBasement009-RightStairs/FCBasement010-TopLeftStairs", new MapLocation(MAP_FCBASEMENT009, 15, 1), new Warp(MAP_FCBASEMENT010, 1, 5, D_RIGHT), false);
            
            defineWarp("FCBasement010-TopLeftStairs/FCBasement009-RightStairs", new MapLocation(MAP_FCBASEMENT010, 3, 1), new Warp(MAP_FCBASEMENT009, 18, 11, D_LEFT), false);
            defineWarp("FCBasement010-RightStairs/FCBasement015-LeftStairs", new MapLocation(MAP_FCBASEMENT010, 14, 1), new Warp(MAP_FCBASEMENT015, 1, 5, D_RIGHT), false);
            defineWarp("FCBasement010-BottomStairs/FCBasement016-Stairs", new MapLocation(MAP_FCBASEMENT010, 15, 1), new Warp(MAP_FCBASEMENT016, 18, 5, D_LEFT), false);
            
            defineWarp("FCBasement011-Door/FCBasement008-LeftDoor", new MapLocation(MAP_FCBASEMENT011, 1, 1), new Warp(MAP_FCBASEMENT008, 7, 8, D_DOWN), false);
            defineWarp("FCBasement011-LeftStairs/FCBasement012-TopStairs", new MapLocation(MAP_FCBASEMENT011, 23, 1), new Warp(MAP_FCBASEMENT012, 15, 1, D_LEFT), false);
            defineWarp("FCBasement011-RightStairsBottom/FCBasement012-RightStairs", new MapLocation(MAP_FCBASEMENT011, 3, 1), new Warp(MAP_FCBASEMENT012, 18, 7, D_LEFT), false);
            defineWarp("FCBasement011-RightStairsTop/FCBasement014-StairsBottom", new MapLocation(MAP_FCBASEMENT011, 13, 1), new Warp(MAP_FCBASEMENT014, 4, 13, D_RIGHT), false);
            
            defineWarp("FCBasement012-Left/FCBasement009-Right", new MapLocation(MAP_FCBASEMENT012, 3, 1), new Warp(MAP_FCBASEMENT009, 19, 7, D_LEFT), false);
            defineWarp("FCBasement012-TopStairs/FCBasement011-LeftStairs", new MapLocation(MAP_FCBASEMENT012, 16, 1), new Warp(MAP_FCBASEMENT011, 1, 8, D_RIGHT), false);
            defineWarp("FCBasement012-RightStairs/FCBasement011-RightStairsBottom", new MapLocation(MAP_FCBASEMENT012, 20, 1), new Warp(MAP_FCBASEMENT011, 9, 13, D_RIGHT), false);
            defineWarp("FCBasement012-BottomStairs/FCBasement015-RightStairs", new MapLocation(MAP_FCBASEMENT012, 21, 1), new Warp(MAP_FCBASEMENT015, 17, 1, D_LEFT), false);
            
            defineWarp("FCBasement013-Stairs/FCBasement009-LeftStairs", new MapLocation(MAP_FCBASEMENT013, 3, 1), new Warp(MAP_FCBASEMENT009, 1, 5, D_RIGHT), false);
            defineWarp("FCBasement013-Door/FCBasement020-Door", new MapLocation(MAP_FCBASEMENT013, 10, 1), new Warp(MAP_FCBASEMENT020, 4, 11, D_DOWN), false);
            defineWarp("FCBasement013-Left/FCBasement008-Right", new MapLocation(MAP_FCBASEMENT013, 13, 1), new Warp(MAP_FCBASEMENT008, 19, 8, D_LEFT), false);
            
            defineWarp("FCBasement014-StairsTop/FCBasement017-Stairs", new MapLocation(MAP_FCBASEMENT014, 14, 1), new Warp(MAP_FCBASEMENT017, 3, 13, D_RIGHT), false);
            defineWarp("FCBasement014-StairsBottom/FCBasement011-RightStairsTop", new MapLocation(MAP_FCBASEMENT014, 15, 1), new Warp(MAP_FCBASEMENT011, 18, 4, D_LEFT), false);
            
            defineWarp("FCBasement015-LeftStairs/FCBasement010-RightStairs", new MapLocation(MAP_FCBASEMENT015, 3, 1), new Warp(MAP_FCBASEMENT010, 18, 11, D_LEFT), false);
            defineWarp("FCBasement015-RightStairs/FCBasement012-BottomStairs", new MapLocation(MAP_FCBASEMENT015, 22, 1), new Warp(MAP_FCBASEMENT012, 5, 13, D_RIGHT), false);
            
            defineWarp("FCBasement016-Stairs/FCBasement010-BottomStairs", new MapLocation(MAP_FCBASEMENT016, 16, 1), new Warp(MAP_FCBASEMENT010, 5, 13, D_RIGHT), false);
            defineWarp("FCBasement016-Door/FCBasement019", new MapLocation(MAP_FCBASEMENT016, 13, 1), new Warp(MAP_FCBASEMENT019, 10, 8, D_DOWN), false);
            
            defineWarp("FCBasement017-Stairs/FCBasement014-StairsTop", new MapLocation(MAP_FCBASEMENT017, 10, 1), new Warp(MAP_FCBASEMENT014, 16, 1, D_LEFT), false);
            defineWarp("FCBasement017-Door/FCBasement018", new MapLocation(MAP_FCBASEMENT017, 1, 1), new Warp(MAP_FCBASEMENT018, 14, 6, D_DOWN), false);
            
            defineWarp("FCBasement018/FCBasement017-Door", new MapLocation(MAP_FCBASEMENT018, 2, 1), new Warp(MAP_FCBASEMENT017, 8, 8, D_DOWN), false);
            
            defineWarp("FCBasement019/FCBasement016-Door", new MapLocation(MAP_FCBASEMENT019, 2, 1), new Warp(MAP_FCBASEMENT016, 4, 9, D_DOWN), false);
            
            defineWarp("FCBasement020-Door/FCBasement013-Door", new MapLocation(MAP_FCBASEMENT020, 1, 1), new Warp(MAP_FCBASEMENT013, 8, 8, D_DOWN), false);
            defineWarp("FCBasement020-Stairs/FCBasement022-BottomStairs", new MapLocation(MAP_FCBASEMENT020, 14, 1), new Warp(MAP_FCBASEMENT022, 4, 13, D_RIGHT), false);
            
            defineWarp("FCBasement021-StairsBottom/FCBasement022-RightStiars", new MapLocation(MAP_FCBASEMENT021, 15, 1), new Warp(MAP_FCBASEMENT022, 18, 5, D_LEFT), false);
            defineWarp("FCBasement021-StairsTop/FCVillage001-Confined", new MapLocation(MAP_FCBASEMENT021, 14, 1), new Warp(MAP_FCDUNGEONVILLAGE, 9, 4, 1, "Confined"));
            
            defineWarp("FCBasement022-BottomStairs/FCBasement020-Stairs", new MapLocation(MAP_FCBASEMENT022, 15, 1), new Warp(MAP_FCBASEMENT020, 16, 1, D_LEFT), false);
            defineWarp("FCBasement022-RightStairs/FCBasement021-StairsBottom", new MapLocation(MAP_FCBASEMENT022, 24, 1), new Warp(MAP_FCBASEMENT021, 4, 13, D_RIGHT), false);
            
            defineWarp("FCSmallMap001-Cave/FCEntranceB", new MapLocation(MAP_FCCAVEMOUTH, 1, 1), new Warp(MAP_FCENTRANCEB, 9, 7, D_DOWN));
            defineWarp("FCSmallMap001/FCFull-SmallMap1", new MapLocation(MAP_FCCAVEMOUTH, 2, 1), new Warp(MAP_FCOVERWORLD, 33, 37, 2, "FC World A"));
            
            defineWarp("FCSmallMap002/FCBasement019", new MapLocation(MAP_FCPYRAMIDS, 1, 1), new Warp(MAP_FCBASEMENT019, 10, 10, D_DOWN));
            defineWarp("FCSmallMap002/FCFull-SmallMap2", new MapLocation(MAP_FCPYRAMIDS, 2, 1), new Warp(MAP_FCOVERWORLD, 25, 143, 2, "FC World B"));
            
            defineWarp("FCVillage001/FCDungeon001", new MapLocation(MAP_FCDUNGEONVILLAGE, 7, 1), new Warp(MAP_FCDUNGEON001, 6, 5, D_DOWN));
            defineWarp("FCVillage001-Confined/FCBasement021-StairsTop", new MapLocation(MAP_FCDUNGEONVILLAGE, 1, 1, "Confined"), new Warp(MAP_FCBASEMENT021, 16, 1, D_LEFT));
            defineWarp("FCVillage001/FCFull-Village1", new MapLocation(MAP_FCDUNGEONVILLAGE, 2, 1), new Warp(MAP_FCOVERWORLD, 34, 90, 2, "FC World A"));
            
            defineWarp("FCDungeon001/FCVillage001", new MapLocation(MAP_FCDUNGEON001, 7, 1), new Warp(MAP_FCDUNGEONVILLAGE, 23, 12, D_DOWN));
            if (!derandomizeEvents || !fixedGlitchEvent) // Either randomly possible to access normal version, or predetermined to always be normal version
                defineWarp("FCDungeon001/FCDungeon002", new MapLocation(MAP_FCDUNGEON001, 26, 1, 2), new Warp(MAP_FCDUNGEON002, 15, 8, D_DOWN));
            if (!derandomizeEvents || fixedGlitchEvent) // Either randomly possible to access glitch version, or predetermined to always be glitch version
                defineWarp("FCDungeon001/FCDungeon002?", new MapLocation(MAP_FCDUNGEON001, 26, 1), new Warp(MAP_FCDUNGEON002GLITCH, 15, 8, D_DOWN));
            defineWarp("FCDungeon001/FCDungeon003", new MapLocation(MAP_FCDUNGEON001, 25, 1), new Warp(MAP_FCDUNGEON003, 15, 8, D_DOWN));
            
            if (!derandomizeEvents || !fixedGlitchEvent) // Either randomly possible to access normal version, or fixed to always be normal version
                defineWarp("FCDungeon002/FCDungeon001", new MapLocation(MAP_FCDUNGEON002, 7, 1), new Warp(MAP_FCDUNGEON001, 120, 65, D_DOWN));
            
            if (!derandomizeEvents || fixedGlitchEvent) // Either randomly possible to access glitch version, or fixed to always be glitch version
                defineWarp("FCDungeon002?/FCDungeon001", new MapLocation(MAP_FCDUNGEON002GLITCH, 87, 1), new Warp(MAP_FCDUNGEON001, 120, 65, D_DOWN));
            
            defineWarp("FCDungeon003/FCDungeon001", new MapLocation(MAP_FCDUNGEON003, 1, 1), new Warp(MAP_FCDUNGEON001, 33, 77, D_DOWN));
            defineWarp("FCDungeon003/Mono012-Stairs", new MapLocation(MAP_FCDUNGEON003, 2, 1), new Warp(MAP_MONO012, 19, 11, D_RIGHT));
            
            defineWarp("FCVillage002/FCVillageUnderground", new MapLocation(MAP_FCLIZARDVILLAGE, 4, 1), new Warp(MAP_FCUNDERVILLAGE, 14, 7, D_DOWN));
            defineWarp("FCVillage002/FCFull-Village2", new MapLocation(MAP_FCLIZARDVILLAGE, 3, 1), new Warp(MAP_FCOVERWORLD, 54, 179, 2, "FC World B"));
            
            defineWarp("FCVillageUnderground/FCVillage002", new MapLocation(MAP_FCUNDERVILLAGE, 1, 1), new Warp(MAP_FCLIZARDVILLAGE, 14, 3, D_DOWN));
            
            defineWarp("FCVillage003/FCFull-Village3", new MapLocation(MAP_FCPIRORIVILLAGE, 2, 1), new Warp(MAP_FCOVERWORLD, 85, 152, 2, "FC World B"));
            
            defineWarp("FCFull-SmallMap1/FCSmallMap001", new MapLocation(MAP_FCOVERWORLD, 1, 1, "FC World A"), new Warp(MAP_FCCAVEMOUTH, 9, 14, D_UP));
            defineWarp("FCFull-Village1/FCVillage001", new MapLocation(MAP_FCOVERWORLD, 2, 1, "FC World A"), new Warp(MAP_FCDUNGEONVILLAGE, 9, 24, D_UP));
            defineWarp("FCFull-SmallMap2/FCSmallMap002", new MapLocation(MAP_FCOVERWORLD, 3, 1, "FC World B"), new Warp(MAP_FCPYRAMIDS, 10, 14, D_UP));
            defineWarp("FCFull-Village2/FCVillage002", new MapLocation(MAP_FCOVERWORLD, 4, 1, "FC World B"), new Warp(MAP_FCLIZARDVILLAGE, 14, 29, D_UP));
            defineWarp("FCFull-Village3/FCVillage003", new MapLocation(MAP_FCOVERWORLD, 6, 1, "FC World B"), new Warp(MAP_FCPIRORIVILLAGE, 7, 14, D_UP));
            
            // Define special groups of exits for reference.
            nexusExitKeys = new string[] { "DoorRoom/01Frog", "DoorRoom/02HatAndScarf", "DoorRoom/03Umbrella",
                                           "DoorRoom/04Knife", "DoorRoom/05YukiOnna", "DoorRoom/06Hairstyle",
                                           "DoorRoom/07Bicycle", "DoorRoom/08EyeballHand", "DoorRoom/09Small",
                                           "DoorRoom/10CatEars", "DoorRoom/11Neon", "DoorRoom/12Lamp" };
            
            backToNexusKeys = new string[] { "01Frog/DoorRoom", "02HatAndScarf/DoorRoom", "03Umbrella/DoorRoom",
                                             "04Knife/DoorRoom", "05YukiOnna/DoorRoom", "06Hairstyle/DoorRoom",
                                             "07Bicycle/DoorRoom", "08EyeballHand/DoorRoom", "09Small/DoorRoom",
                                             "10CatEars/DoorRoom", "11Neon/DoorRoom", "12Lamp/DoorRoom" };
            
            intoHellKeys = new string[] { "Thorns-RedBoxes/RedA-Thorns", "FootprintsB-RedBoxes/RedA-Footprints", "CheckerB-RedBoxes/RedA-Checker",
                                          "NeonTilesB-RedBoxes/RedA-NeonTiles", "Jukai2-Gate/RedA-JukaiGate", "LakeCorridor-RightSide/RedA-LakeStairs" };
            
            outtaHellKeys = new string[] { "RedA-Thorns/Thorns-RedBoxes", "RedA-Footprints/FootprintsB-RedBoxes", "RedA-Checker/CheckerB-RedBoxes",
                                           "RedA-NeonTiles/NeonTilesB-RedBoxes", "RedA-JukaiGate/Jukai2-Gate", "RedA-LakeStairs/LakeCorridor-RightSide" };
        }
        
        // Puts a CommandLocation in warpLocations, and a Warp in exits. Also adds to list of non-shuffled warps if specified.
        static void defineWarp(string name, CommandLocation location, Warp warp, bool doShuffle = true)
        {
            if (warpLocations.ContainsKey(name))
                debugMessage("Warning: Already defined warp \"" + name + "\"!");
            
            warpLocations[name] = location;
            exits[name] = warp;
            
            if (!doShuffle)
                noShuffleKeys.Add(name);
        }
        
        // Defines which warps are connected to each other.
        static void initExitLinks()
        {
            linkedExits = new List<WarpLink>();
            
            if (derandomizeEvents) // Without random element, dream bed and staircase can be considered a proper link
            {
                string bedWarpKey = new string[] { "StairsInBed-Upstairs/DreamRoom",
                                                   "StairsInBed-Upstairs/02HatAndScarf",
                                                   "StairsInBed-Upstairs/05YukiOnna",
                                                   "StairsInBed-Upstairs/09Small",
                                                   "StairsInBed-Upstairs/LampSmallRoom2" }[fixedDreamBedID - 1];
                defineExitLink("DreamBedCommon/StairsInBed", bedWarpKey);
            }
            
            defineExitLink("RealWorldRoom/FamiGame1", "FamiGame1/RealWorldRoom");
            
            defineExitLink("DreamRoom/DoorRoom", "DoorRoom/DreamRoom");
            
            defineExitLink("DoorRoom/01Frog", "01Frog/DoorRoom");
            defineExitLink("DoorRoom/02HatAndScarf", "02HatAndScarf/DoorRoom");
            defineExitLink("DoorRoom/03Umbrella", "03Umbrella/DoorRoom");
            defineExitLink("DoorRoom/04Knife", "04Knife/DoorRoom");
            defineExitLink("DoorRoom/05YukiOnna", "05YukiOnna/DoorRoom");
            defineExitLink("DoorRoom/06Hairstyle", "06Hairstyle/DoorRoom");
            defineExitLink("DoorRoom/07Bicycle", "07Bicycle/DoorRoom");
            defineExitLink("DoorRoom/08EyeballHand", "08EyeballHand/DoorRoom");
            defineExitLink("DoorRoom/09Small", "09Small/DoorRoom");
            defineExitLink("DoorRoom/10CatEars", "10CatEars/DoorRoom");
            defineExitLink("DoorRoom/11Neon", "11Neon/DoorRoom");
            defineExitLink("DoorRoom/12Lamp", "12Lamp/DoorRoom");
            
            defineExitLink("01Frog-GateDown/Thorns-Top", "Thorns-Top/01Frog-GateDown");
            defineExitLink("01Frog-GateUp/Thorns-Bottom", "Thorns-Bottom/01Frog-GateUp");
            
            defineExitLink("02HatAndScarf-Gate/Mono-Mouth", "Mono-Mouth/02HatAndScarf-Gate");
            
            defineExitLink("03Umbrella-Subway/Highway-Subway", "Highway-Subway/03Umbrella-Subway");
            
            defineExitLink("04Knife-Gate/Sand-Gate", "Sand-Gate/04Knife-Gate");
            
            defineExitLink("05YukiOnna/Igloo1", "Igloo1/05YukiOnna");
            defineExitLink("05YukiOnna/Igloo2", "Igloo2/05YukiOnna");
            defineExitLink("05YukiOnna/Igloo3", "Igloo3/05YukiOnna");
            defineExitLink("05YukiOnna/Igloo4", "Igloo4/05YukiOnna");
            defineExitLink("05YukiOnna/Igloo5", "Igloo5/05YukiOnna");
            defineExitLink("05YukiOnna/Igloo6", "Igloo6/05YukiOnna");
            defineExitLink("05YukiOnna/Igloo7", "Igloo7/05YukiOnna");
            
            defineExitLink("06Hairstyle-Manhole/Gutter_001-Ladder", "Gutter_001-Ladder/06Hairstyle-Manhole");
            
            defineExitLink("07Bicycle-Elevator/DepartmentA-Elevator", "DepartmentA-Elevator/07Bicycle-Elevator");
            
            defineExitLink("08EyeballHand-Mouth/FootprintsA", "FootprintsA/08EyeballHand-Mouth");
            
            defineExitLink("09Small-Pyramid/CheckerA-Pyramid", "CheckerA-Pyramid/09Small-Pyramid");
            
            defineExitLink("10CatEars-Gate/FCEntranceA-Gate", "FCEntranceA-Gate/10CatEars-Gate");
            
            defineExitLink("11Neon-Door/NeonTilesA-Pyramid", "NeonTilesA-Pyramid/11Neon-Door");
            
            defineExitLink("12Lamp-GuardedDoor/LampSmallRoom1", "LampSmallRoom1/12Lamp-GuardedDoor");
            defineExitLink("12Lamp-LeftDoor/LampSmallRoom2-Door", "LampSmallRoom2-Door/12Lamp-LeftDoor");
            defineExitLink("12Lamp-Horseshoe/LampSmallRoom3-LonelyDoor", "LampSmallRoom3-LonelyDoor/12Lamp-Horseshoe");
            defineExitLink("12Lamp-CanalDoor/LampSmallRoom3-WallDoor", "LampSmallRoom3-WallDoor/12Lamp-CanalDoor");
            defineExitLink("12Lamp-Zipper/LampSmallRoom4", "LampSmallRoom4/12Lamp-Zipper");
            
            defineExitLink("Igloo7-Pond/Shallows", "Shallows/Igloo7-Pond");
            
            defineExitLink("LampSmallRoom3-SmallLamp/CheckerA-Isolated", "CheckerA-Isolated/LampSmallRoom3-SmallLamp");
            
            defineExitLink("LampSmallRoom4/LampSmallRoom5", "LampSmallRoom5/LampSmallRoom4");
            
            defineExitLink("NeonTilesA/NeonTilesB", "NeonTilesB/NeonTilesA");
            
            defineExitLink("CheckerA/CheckerB", "CheckerB/CheckerA");
            
            if (heckificationType != 2) // Don't include Hell links as candidates if the goal is to remove Hell from the game
            {
                defineExitLink("RedA-Thorns/Thorns-RedBoxes", "Thorns-RedBoxes/RedA-Thorns");
                defineExitLink("RedA-Footprints/FootprintsB-RedBoxes", "FootprintsB-RedBoxes/RedA-Footprints");
                defineExitLink("RedA-Checker/CheckerB-RedBoxes", "CheckerB-RedBoxes/RedA-Checker");
                defineExitLink("RedA-NeonTiles/NeonTilesB-RedBoxes", "NeonTilesB-RedBoxes/RedA-NeonTiles");
                defineExitLink("RedA-JukaiGate/Jukai2-Gate", "Jukai2-Gate/RedA-JukaiGate");
                defineExitLink("RedA-LakeStairs/LakeCorridor-RightSide", "LakeCorridor-RightSide/RedA-LakeStairs");
            }
            
            defineExitLink("FootprintsA/FootprintsB", "FootprintsB/FootprintsA");
            
            defineExitLink("Sand-Left/SandH-Right", "SandH-Right/Sand-Left");
            defineExitLink("Sand-Right/SandB-Left", "SandB-Left/Sand-Right");
            
            defineExitLink("SandA-Left/SandE-Party", "SandE-Party/SandA-Left");
            
            defineExitLink("SandA-Right/SandH-Left", "SandH-Left/SandA-Right");
            
            defineExitLink("SandB-UpRight/SandC-Down", "SandC-Down/SandB-UpRight");
            defineExitLink("SandB-UpLeft/SandF-Down", "SandF-Down/SandB-UpLeft");
            
            defineExitLink("SandC-UpRight/SandD-Down", "SandD-Down/SandC-UpRight");
            defineExitLink("SandC-UpLeft/SandG-Down", "SandG-Down/SandC-UpLeft");
            
            defineExitLink("SandF-Fence/Earth-Fence", "Earth-Fence/SandF-Fence");
            
            defineExitLink("SandG-Box/FCEntranceB", "FCEntranceB/SandG-Box");
            defineExitLink("SandG-Left/SandK-Right", "SandK-Right/SandG-Left");
            
            defineExitLink("SandH-Passage/SandJ-Passage", "SandJ-Passage/SandH-Passage");
            
            defineExitLink("SandI-Passage/SandJ-Passage", "SandJ-Passage/SandI-Passage");
            defineExitLink("SandI-Right/SandK-Left", "SandK-Left/SandI-Right");
            defineExitLink("SandI-Up/SandL-Down", "SandL-Down/SandI-Up");
            
            defineExitLink("SandJ-Shack/InsideShackA", "InsideShackA/SandJ-Shack");
            
            defineExitLink("SandL-Right/SandM-Left", "SandM-Left/SandL-Right");
            
            defineExitLink("SandM-Stairs/SandStairs", "SandStairs/SandM-Stairs");
            
            defineExitLink("SandStairs/Park-Stairs", "Park-Stairs/SandStairs");
            
            defineExitLink("Earth-UpperHouse/EarthA-RightDoor", "EarthA-RightDoor/Earth-UpperHouse");
            defineExitLink("Earth-LowerHouse/EarthB", "EarthB/Earth-LowerHouse");
            
            defineExitLink("Gutter_001-Left/Gutter_002-Right", "Gutter_002-Right/Gutter_001-Left");
            
            defineExitLink("Gutter_002-Left/Gutter_003-Right", "Gutter_003-Right/Gutter_002-Left");
            
            defineExitLink("Gutter_003-Up/Gutter_004-Left", "Gutter_004-Left/Gutter_003-Up");
            defineExitLink("Gutter_003-Left/Gutter_006-Right", "Gutter_006-Right/Gutter_003-Left");
            
            defineExitLink("Gutter_004-SmallOpening/Gutter_005-Left", "Gutter_005-Left/Gutter_004-SmallOpening"); // Need Small to enter opening
            defineExitLink("Gutter_004-Right/RedRoom", "RedRoom/Gutter_004-Right");
            
            defineExitLink("Gutter_005-Stairs/LakeCorridor-ByVending", "LakeCorridor-ByVending/Gutter_005-Stairs");
            
            defineExitLink("DepartmentA-DownEscalator/DepartmentB-UpEscalator", "DepartmentB-UpEscalator/DepartmentA-DownEscalator");
            defineExitLink("DepartmentA-UpEscalator/DepartmentD-Escalator", "DepartmentD-Escalator/DepartmentA-UpEscalator");
            
            defineExitLink("DepartmentB-DownEscalator/DepartmentC-UpEscalator", "DepartmentC-UpEscalator/DepartmentB-DownEscalator");
            defineExitLink("DepartmentB-LeftDoor/EntranceA", "EntranceA/DepartmentB-LeftDoor");
            defineExitLink("DepartmentB-RightDoor/EntranceC", "EntranceC/DepartmentB-RightDoor");
            
            defineExitLink("DepartmentC-Hole/Jukai1-Hole", "Jukai1-Hole/DepartmentC-Hole");
            
            defineExitLink("EntranceA/EntranceB", "EntranceB/EntranceA");
            
            defineExitLink("Jukai1/Jukai2-Left", "Jukai2-Left/Jukai1");
            
            defineExitLink("Jukai2-Right/Jukai3-Top", "Jukai3-Top/Jukai2-Right");
            defineExitLink("Jukai2-WormHole/Mono010-LeftWorm", "Mono010-LeftWorm/Jukai2-WormHole");
            
            defineExitLink("Jukai3-Bottom/JukaiTrain", "JukaiTrain/Jukai3-Bottom");
            
            defineExitLink("JukaiTrain/TrainInterior-PreTrip", "TrainInterior-PreTrip/JukaiTrain");
            
            defineExitLink("TrainInterior-PostTrip/JukaiTrainB", "JukaiTrainB/TrainInterior-PostTrip");
            
            defineExitLink("JukaiTrainB/Jukai4", "Jukai4/JukaiTrainB");
            
            defineExitLink("Jukai4/Bridge", "Bridge/Jukai4");
            
            defineExitLink("Bridge/Shore", "Shore/Bridge");
            
            defineExitLink("Mono-TunnelLeft/Mono001-Left", "Mono001-Left/Mono-TunnelLeft");
            defineExitLink("Mono-TunnelRight/Mono001-Right", "Mono001-Right/Mono-TunnelRight");
            defineExitLink("Mono-LoopTunnelRight/Mono002-Right", "Mono002-Right/Mono-LoopTunnelRight"); // Need Knife to enter loop tunnel
            defineExitLink("Mono-LoopTunnelLeft/Mono003-TunnelRight", "Mono003-TunnelRight/Mono-LoopTunnelLeft"); // Need Knife to enter loop tunnel
            defineExitLink("Mono-EyeBox/Mono005", "Mono005/Mono-EyeBox");
            
            defineExitLink("Mono002-Left/Mono006-Right", "Mono006-Right/Mono002-Left");
            
            defineExitLink("Mono004/Mono006-Left", "Mono006-Left/Mono004");
            
            defineExitLink("Mono008/Mono009", "Mono009/Mono008");
            defineExitLink("Mono008-Left/Mono011-Head", "Mono011-Head/Mono008-Left");
            
            defineExitLink("Mono010-RightWorm/Mono011-WormHole", "Mono011-WormHole/Mono010-RightWorm");
            
            defineExitLink("Mono011-Stairs/Mono013-Stairs", "Mono013-Stairs/Mono011-Stairs");
            
            defineExitLink("Mono012/Mono013", "Mono013/Mono012");
            defineExitLink("Mono012-Stairs/FCDungeon003", "FCDungeon003/Mono012-Stairs");
            
            defineExitLink("Shallows/ShallowsHouse", "ShallowsHouse/Shallows");
            
            defineExitLink("Park/ParkCliff", "ParkCliff/Park");
            
            defineExitLink("ParkCliff/InsideBricksA", "InsideBricksA/ParkCliff");
            
            defineExitLink("Wall/InsideBricksC", "InsideBricksC/Wall");
            
            defineExitLink("InsideBricksA-Darkness/InsideBricksB-Normal", "InsideBricksB-Normal/InsideBricksA-Darkness");
            
            defineExitLink("InsideBricksB-Flower/InsideBricksC-Darkness", "InsideBricksC-Darkness/InsideBricksB-Flower");
            
            defineExitLink("FCEntranceA-Ladder/FCHouse-Ladder", "FCHouse-Ladder/FCEntranceA-Ladder");
            
            defineExitLink("FCHouse/FCInsideHouse", "FCInsideHouse/FCHouse");
            
            defineExitLink("FCInsideHouse/FCBasement001-Stairs", "FCBasement001-Stairs/FCInsideHouse");
            
            defineExitLink("FCBasement001-RightDoor/FCBasement002-UpstairsDoor", "FCBasement002-UpstairsDoor/FCBasement001-RightDoor");
            defineExitLink("FCBasement001-LeftDoor/FCBasement003-LeftDoor", "FCBasement003-LeftDoor/FCBasement001-LeftDoor");
            
            defineExitLink("FCBasement002-RightDoor/FCBasement003-RightDoor", "FCBasement003-RightDoor/FCBasement002-RightDoor");
            defineExitLink("FCBasement002-LeftDoor/FCBasement006-Door", "FCBasement006-Door/FCBasement002-LeftDoor");
            
            defineExitLink("FCBasement003-MiddleDoor/FCBasement004-Right", "FCBasement004-Right/FCBasement003-MiddleDoor");
            
            defineExitLink("FCBasement004-Left/FCBasement005-LeftDoor", "FCBasement005-LeftDoor/FCBasement004-Left");
            
            defineExitLink("FCBasement005-RightDoor/FCBasement007-RightDoor", "FCBasement007-RightDoor/FCBasement005-RightDoor");
            
            defineExitLink("FCBasement006-Right/FCBasement007-Left", "FCBasement007-Left/FCBasement006-Right");
            defineExitLink("FCBasement006-Stairs/FCBasement008-Stairs", "FCBasement008-Stairs/FCBasement006-Stairs");
            
            defineExitLink("FCBasement007-LeftDoor/FCBasement008-RightDoor", "FCBasement008-RightDoor/FCBasement007-LeftDoor");
            
            defineExitLink("FCBasement008-LeftDoor/FCBasement011-Door", "FCBasement011-Door/FCBasement008-LeftDoor");
            defineExitLink("FCBasement008-Right/FCBasement013-Left", "FCBasement013-Left/FCBasement008-Right");
            
            defineExitLink("FCBasement009-RightStairs/FCBasement010-TopLeftStairs", "FCBasement010-TopLeftStairs/FCBasement009-RightStairs");
            defineExitLink("FCBasement009-Right/FCBasement012-Left", "FCBasement012-Left/FCBasement009-Right");
            defineExitLink("FCBasement009-LeftStairs/FCBasement013-Stairs", "FCBasement013-Stairs/FCBasement009-LeftStairs");
            
            defineExitLink("FCBasement010-RightStairs/FCBasement015-LeftStairs", "FCBasement015-LeftStairs/FCBasement010-RightStairs");
            defineExitLink("FCBasement010-BottomStairs/FCBasement016-Stairs", "FCBasement016-Stairs/FCBasement010-BottomStairs");
            
            defineExitLink("FCBasement011-LeftStairs/FCBasement012-TopStairs", "FCBasement012-TopStairs/FCBasement011-LeftStairs");
            defineExitLink("FCBasement011-RightStairsBottom/FCBasement012-RightStairs", "FCBasement012-RightStairs/FCBasement011-RightStairsBottom");
            defineExitLink("FCBasement011-RightStairsTop/FCBasement014-StairsBottom", "FCBasement014-StairsBottom/FCBasement011-RightStairsTop");
            
            defineExitLink("FCBasement012-BottomStairs/FCBasement015-RightStairs", "FCBasement015-RightStairs/FCBasement012-BottomStairs");
            
            defineExitLink("FCBasement013-Door/FCBasement020-Door", "FCBasement020-Door/FCBasement013-Door");
            
            defineExitLink("FCBasement014-StairsTop/FCBasement017-Stairs", "FCBasement017-Stairs/FCBasement014-StairsTop");
            
            defineExitLink("FCBasement016-Door/FCBasement019", "FCBasement019/FCBasement016-Door");
            
            defineExitLink("FCBasement017-Door/FCBasement018", "FCBasement018/FCBasement017-Door");
            
            defineExitLink("FCBasement020-Stairs/FCBasement022-BottomStairs", "FCBasement022-BottomStairs/FCBasement020-Stairs");
            
            defineExitLink("FCBasement021-StairsBottom/FCBasement022-RightStiars", "FCBasement022-RightStairs/FCBasement021-StairsBottom");
            defineExitLink("FCBasement021-StairsTop/FCVillage001-Confined", "FCVillage001-Confined/FCBasement021-StairsTop");
            
            defineExitLink("FCEntranceB/FCSmallMap001-Cave", "FCSmallMap001-Cave/FCEntranceB");
            
            defineExitLink("FCSmallMap001/FCFull-SmallMap1", "FCFull-SmallMap1/FCSmallMap001");
            
            defineExitLink("FCSmallMap002/FCFull-SmallMap2", "FCFull-SmallMap2/FCSmallMap002");
            
            defineExitLink("FCVillage001/FCDungeon001", "FCDungeon001/FCVillage001");
            defineExitLink("FCVillage001/FCFull-Village1", "FCFull-Village1/FCVillage001");
            
            if (!derandomizeEvents || !fixedGlitchEvent) // Either randomly possible to access normal version, or fixed to always be normal version
                defineExitLink("FCDungeon001/FCDungeon002", "FCDungeon002/FCDungeon001");
            if (!derandomizeEvents || fixedGlitchEvent) // Either randomly possible to access glitch version, or fixed to always be glitch version
                defineExitLink("FCDungeon001/FCDungeon002?", "FCDungeon002?/FCDungeon001");
            defineExitLink("FCDungeon001/FCDungeon003", "FCDungeon003/FCDungeon001");
            
            defineExitLink("FCVillage002/FCVillageUnderground", "FCVillageUnderground/FCVillage002");
            defineExitLink("FCVillage002/FCFull-Village2", "FCFull-Village2/FCVillage002");
            
            defineExitLink("FCVillage003/FCFull-Village3", "FCFull-Village3/FCVillage003");
            
            defineExitLink("StairsInBed-Downstairs/PassageOutsideStorage", "PassageOutsideStorage/StairsInBed-Downstairs");
            
            defineExitLink("PassageOutsideStorage-Left/OutsideStorage(Flames)", "OutsideStorage(Flames)/PassageOutsideStorage-Left");
            
            defineExitLink("OutsideStorage(Flames)/Storeroom", "Storeroom/OutsideStorage(Flames)");
            
            defineExitLink("Storeroom/SpaceshipEntrance", "SpaceshipEntrance/Storeroom");
            
            defineExitLink("Spaceship_01-LeftFlying/Spaceship_02", "Spaceship_02/Spaceship_01-LeftFlying");
            defineExitLink("Spaceship_01-Right/Spaceship_03", "Spaceship_03/Spaceship_01-Right");
            defineExitLink("Spaceship_01-LeftLanded/SpaceshipExit", "SpaceshipExit/Spaceship_01-LeftLanded");
            
            defineExitLink("SpaceshipExit/Mars_01", "Mars_01/SpaceshipExit");
            
            defineExitLink("Mars_01/Mars_02", "Mars_02/Mars_01");
            
            defineExitLink("Mars_02/Mars_03", "Mars_03/Mars_02");
            
            defineExitLink("Mars_03/Mars_04", "Mars_04/Mars_03");
            
            defineExitLink("Mars_04/Mars_05", "Mars_05/Mars_04");
            
            defineExitLink("Mars_05-Small/MarsBasement_01", "MarsBasement_01/Mars_05-Small");
            
            defineExitLink("MarsBasement_01/MarsBasement_02", "MarsBasement_02/MarsBasement_01");
            
            defineExitLink("DepartmentD/DepartmentRoof", "DepartmentRoof/DepartmentD");
            
            defineExitLink("DepartmentRoof/FlyWithBroom", "FlyWithBroom/DepartmentRoof");
        }
        
        // Defines two warps as being connected.
        static void defineExitLink(string warp1, string warp2)
        {
            if (!warpLocations.ContainsKey(warp1))
            {
                debugMessage("Warning: Link definition refers to non-existent warp " + warp1 + "!");
                return;
            }
            if (!warpLocations.ContainsKey(warp2))
            {
                debugMessage("Warning: Link definition refers to non-existent warp " + warp2 + "!");
                return;
            }
            
            linkedExits.Add(new WarpLink(warp1, warp2));
        }
        
        // Defines display names for each individual map.
        static void initMapNames()
        {
            mapDisplayNames = new string[180];
            
            mapDisplayNames[MAP_TESTMAP] = "Test Map";
            mapDisplayNames[MAP_GAMESTART] = "Game Start";
            mapDisplayNames[MAP_REALWORLDROOM] = "Real-World Room";
            mapDisplayNames[MAP_VERANDA] = "Veranda";
            // 5: N/A
            mapDisplayNames[MAP_DREAMROOM] = "Dream Room";
            mapDisplayNames[MAP_DREAMVERANDAA] = "Dream Veranda A";
            mapDisplayNames[MAP_DREAMVERANDAB] = "Dream Veranda B";
            mapDisplayNames[MAP_NEXUS] = "Nexus";
            mapDisplayNames[MAP_FORESTWORLD] = "Forest World";
            mapDisplayNames[MAP_BLOCKWORLD] = "Block World";
            mapDisplayNames[MAP_PUDDLEWORLD] = "Puddle World";
            mapDisplayNames[MAP_DARKWORLD] = "Dark World";
            mapDisplayNames[MAP_SNOWWORLD] = "Snow World";
            mapDisplayNames[MAP_MURALWORLD] = "Mural World";
            mapDisplayNames[MAP_GRAFFITIWORLD] = "Graffiti World";
            mapDisplayNames[MAP_EYEBALLWORLD] = "Eyeball World";
            mapDisplayNames[MAP_CANDLEWORLD] = "Candle World";
            mapDisplayNames[MAP_SHIELDFOLKWORLD] = "Shield-Folk World";
            mapDisplayNames[MAP_NEONWORLD] = "Neon World";
            mapDisplayNames[MAP_NUMBERWORLD] = "Number World";
            mapDisplayNames[MAP_IGLOO1] = "Igloo 1";
            mapDisplayNames[MAP_IGLOO2] = "Igloo 2";
            mapDisplayNames[MAP_IGLOO3] = "Igloo 3";
            mapDisplayNames[MAP_IGLOO4] = "Igloo 4";
            mapDisplayNames[MAP_IGLOO5] = "Igloo 5";
            mapDisplayNames[MAP_IGLOO6] = "Igloo 6";
            mapDisplayNames[MAP_IGLOO7] = "Igloo 7";
            mapDisplayNames[MAP_STABBINGROOM] = "Number World: Stabbing Room";
            mapDisplayNames[MAP_ROOMOFBEDS] = "Number World: Bed Room";
            mapDisplayNames[MAP_LAMPWORLD] = "Lamp World";
            mapDisplayNames[MAP_FACESTAIRWAYPASSAGE] = "The Stairway: Passage";
            mapDisplayNames[MAP_FACESTAIRWAY] = "The Stairway";
            mapDisplayNames[MAP_FACE] = "FACE";
            mapDisplayNames[MAP_GUILLOTINEWORLDSMALL] = "Guillotine World (Small)";
            mapDisplayNames[MAP_GUILLOTINEWORLDBIG] = "Guillotine World (Big)";
            mapDisplayNames[MAP_NEONTILESA] = "Neon Tiles Path A";
            mapDisplayNames[MAP_NEONTILESB] = "Neon Tiles Path B";
            mapDisplayNames[MAP_CHECKERTILESA] = "Checker Tiles Path A";
            mapDisplayNames[MAP_CHECKERTILESB] = "Checker Tiles Path B";
            mapDisplayNames[MAP_FACECARPETPLAZA] = "Forest World: Face Carpet Plaza";
            mapDisplayNames[MAP_HIGHWAY] = "Dense Woods A";
            mapDisplayNames[MAP_HIGHWAYA] = "Infinite Road A";
            mapDisplayNames[MAP_HIGHWAYB] = "Infinite Road B";
            mapDisplayNames[MAP_HIGHWAYC] = "Infinite Road C";
            mapDisplayNames[MAP_HIGHWAYD] = "Infinite Road D";
            mapDisplayNames[MAP_HELL] = "Hell";
            mapDisplayNames[MAP_FOOTPRINTPATHA] = "Footprint Path A";
            mapDisplayNames[MAP_FOOTPRINTPATHB] = "Footprint Path B";
            mapDisplayNames[MAP_WILDERNESSGATE] = "Wilderness: Gate";
            mapDisplayNames[MAP_WILDERNESSROUTE4] = "Wilderness: Route 4";
            mapDisplayNames[MAP_WILDERNESSROUTE1] = "Wilderness: Route 1";
            mapDisplayNames[MAP_WILDERNESSROUTE2] = "Wilderness: Route 2";
            mapDisplayNames[MAP_WILDERNESSRAVEBOX] = "Wilderness: Rave Box";
            mapDisplayNames[MAP_WILDERNESSPARTY] = "Wilderness: Party";
            mapDisplayNames[MAP_WILDERNESSTOSETTLEMENT] = "Wilderness: Outside Settlement";
            mapDisplayNames[MAP_WILDERNESSFCPORTAL] = "Wilderness: Fenced FC Portal";
            mapDisplayNames[MAP_WILDERNESSROUTE3] = "Wilderness: Route 3";
            mapDisplayNames[MAP_WILDERNESSGAPINFENCE] = "Wilderness: Gap in the Fence";
            mapDisplayNames[MAP_INFINITEWILDERNESS] = "The Infinite Wilderness";
            mapDisplayNames[MAP_WILDERNESSFENCE] = "Wilderness: Along Fence";
            mapDisplayNames[MAP_WILDERNESSPURPLEPYLONS] = "Wilderness: Purple Pylons";
            mapDisplayNames[MAP_WILDERNESSFOOTOFSTAIRS] = "Wilderness: Foot of the Stairs";
            mapDisplayNames[MAP_STAIRWAYTOTHESKY] = "Stairway to the Sky";
            mapDisplayNames[MAP_INSIDESHACKA] = "Wilderness: Shack";
            mapDisplayNames[MAP_BARRACKSSETTLEMENT] = "Barracks Settlement";
            mapDisplayNames[MAP_MINIHELL] = "Mini Hell";
            mapDisplayNames[MAP_BARRACKSSHACKA] = "Barracks Settlement: Upper Shack";
            mapDisplayNames[MAP_BARRACKSSHACKB] = "Barracks Settlement: Lower Shack";
            mapDisplayNames[MAP_GUTTER001] = "Sewers: Entrance Tunnel";
            mapDisplayNames[MAP_GUTTER002] = "Sewers: City";
            mapDisplayNames[MAP_GUTTER003] = "Sewers: Processing Plant 1";
            mapDisplayNames[MAP_GUTTER004] = "Sewers: Tunnel to Big Red";
            mapDisplayNames[MAP_GUTTER005] = "Sewers: Small Tunnel";
            mapDisplayNames[MAP_GUTTER006] = "Sewers: Processing Plant 2";
            mapDisplayNames[MAP_MALLA] = "The Mall: Upper Floor";
            mapDisplayNames[MAP_MALLB] = "The Mall: Main Floor";
            mapDisplayNames[MAP_MALLRECEPTION] = "The Mall: Reception Desk";
            mapDisplayNames[MAP_MALLMUSIC] = "The Mall: Flute Room";
            mapDisplayNames[MAP_MALLTOKUTO] = "The Mall: Tokuto-kun's Room";
            mapDisplayNames[MAP_BIGRED] = "Big Red";
            mapDisplayNames[MAP_MALLC] = "The Mall: Lower Floor";
            mapDisplayNames[MAP_WINDMILLWORLD] = "Windmill World";
            mapDisplayNames[MAP_JUKAI1] = "Dense Woods B: Entrance";
            // 85: Sub-Maps
            mapDisplayNames[MAP_JUKAI2] = "Dense Woods B";
            mapDisplayNames[MAP_JUKAITRAINBEFORE] = "Outside Traincar (Woods Side)";
            mapDisplayNames[MAP_TRAININTERIOR] = "Inside Traincar";
            mapDisplayNames[MAP_JUKAITRAINAFTER] = "Outside Traincar (Witch's Side)";
            mapDisplayNames[MAP_JUKAI3] = "Dark Woods";
            mapDisplayNames[MAP_JUKAI4] = "Witch's Island: Before Bridge";
            mapDisplayNames[MAP_WITCHBRIDGE] = "Witch's Island: Bridge";
            mapDisplayNames[MAP_WITCHISLAND] = "Witch's Island";
            mapDisplayNames[MAP_MONO] = "White Desert A";
            mapDisplayNames[MAP_MONO001] = "White Desert A: Monoko's Tunnel";
            mapDisplayNames[MAP_MONO002] = "White Desert A: Long Tunnel";
            mapDisplayNames[MAP_MONO003] = "White Desert A: Monoe's Tunnel";
            mapDisplayNames[MAP_MONO004] = "White Desert A: Heads in the Sky";
            mapDisplayNames[MAP_MONO005] = "White Desert A: Eye People";
            mapDisplayNames[MAP_MONO006] = "White Desert A: Body Parts";
            mapDisplayNames[MAP_MONO007] = "Uboa's Trap";
            mapDisplayNames[MAP_MONO008] = "White Desert B: Brain Creature";
            mapDisplayNames[MAP_MONO009] = "White Desert B: Quivering Jaw";
            mapDisplayNames[MAP_MONO010] = "White Desert B: Roary Straw";
            mapDisplayNames[MAP_MONO011] = "White Desert B";
            mapDisplayNames[MAP_MONO012] = "White Desert B: Underground Lagoon";
            mapDisplayNames[MAP_MONO013] = "White Desert B: Underground Stairs";
            mapDisplayNames[MAP_PINKSEA] = "The Pink Sea";
            mapDisplayNames[MAP_PONIKOHOUSE] = "Poniko's House";
            mapDisplayNames[MAP_WARPMAZE] = "Teleporter Maze";
            mapDisplayNames[MAP_DOCKS] = "The Docks";
            mapDisplayNames[MAP_PARK] = "Sky Garden";
            mapDisplayNames[MAP_PARKCLIFF] = "Sky Garden's Edge";
            mapDisplayNames[MAP_GHOSTWORLD] = "Ghost World";
            mapDisplayNames[MAP_INSIDEBRICKSA] = "Crossover Garden: Garden Side";
            mapDisplayNames[MAP_INSIDEBRICKSB] = "Crossover Garden: Flowers";
            mapDisplayNames[MAP_INSIDEBRICKSC] = "Crossover Garden: Ghost Side";
            mapDisplayNames[MAP_FCENTRANCEA] = "Static Maze";
            // 119: FC Maps
            mapDisplayNames[MAP_FCHOUSEOUTSIDE] = "FC World A: Outside House";
            mapDisplayNames[MAP_FCHOUSEINSIDE] = "FC World A: Inside House";
            mapDisplayNames[MAP_FCBASEMENT001] = "FC Basement 001";
            mapDisplayNames[MAP_FCBASEMENT002] = "FC Basement 002";
            mapDisplayNames[MAP_FCBASEMENT003] = "FC Basement 003";
            mapDisplayNames[MAP_FCBASEMENT004] = "FC Basement 004";
            mapDisplayNames[MAP_FCBASEMENT005] = "FC Basement 005";
            mapDisplayNames[MAP_FCBASEMENT006] = "FC Basement 006";
            mapDisplayNames[MAP_FCBASEMENT007] = "FC Basement 007";
            mapDisplayNames[MAP_FCBASEMENT008] = "FC Basement 008";
            mapDisplayNames[MAP_FCBASEMENT009] = "FC Basement 009";
            mapDisplayNames[MAP_FCBASEMENT010] = "FC Basement 010";
            mapDisplayNames[MAP_FCBASEMENT011] = "FC Basement 011";
            mapDisplayNames[MAP_FCBASEMENT012] = "FC Basement 012";
            mapDisplayNames[MAP_FCBASEMENT013] = "FC Basement 013";
            mapDisplayNames[MAP_FCBASEMENT014] = "FC Basement 014";
            mapDisplayNames[MAP_FCBASEMENT015] = "FC Basement 015";
            mapDisplayNames[MAP_FCBASEMENT016] = "FC Basement 016";
            mapDisplayNames[MAP_FCBASEMENT017] = "FC Basement 017";
            mapDisplayNames[MAP_FCBASEMENT018] = "FC Basement 018";
            mapDisplayNames[MAP_FCBASEMENT019] = "FC Basement 019";
            mapDisplayNames[MAP_FCBASEMENT020] = "FC Basement 020";
            mapDisplayNames[MAP_FCBASEMENT021] = "FC Basement 021";
            mapDisplayNames[MAP_FCBASEMENT022] = "FC Basement 022";
            mapDisplayNames[MAP_FCENTRANCEB] = "Static Pathway";
            mapDisplayNames[MAP_FCCAVEMOUTH] = "FC World A: Cave Entrance";
            mapDisplayNames[MAP_FCPYRAMIDS] = "FC World B: Oni Pyramids";
            mapDisplayNames[MAP_FCDUNGEONVILLAGE] = "FC World A: Village";
            mapDisplayNames[MAP_FCDUNGEON001] = "FC Dungeon";
            mapDisplayNames[MAP_FCDUNGEON002] = "FC Dungeon: Empty Room";
            mapDisplayNames[MAP_FCDUNGEON003] = "FC Dungeon: White Desert B Exit";
            mapDisplayNames[MAP_FCLIZARDVILLAGE] = "FC World B: Lizard Village";
            mapDisplayNames[MAP_FCUNDERVILLAGE] = "FC World B: Under-Village";
            mapDisplayNames[MAP_FCSMALLMAP003] = "Unused FC World Map";
            mapDisplayNames[MAP_FCPIRORIVILLAGE] = "FC World B: Pirori Village";
            mapDisplayNames[MAP_NASU] = "NASU";
            mapDisplayNames[MAP_ENDING] = "The End";
            mapDisplayNames[MAP_FCDUNGEON002GLITCH] = "FC Dungeon: Glitch Room";
            mapDisplayNames[MAP_FCOVERWORLD] = "FC Overworld";
            mapDisplayNames[MAP_STAIRCASEOFHANDS] = "Staircase of Hands";
            mapDisplayNames[MAP_BEFOREBLAZINGCORRIDOR] = "Before Blazing Corridor";
            mapDisplayNames[MAP_BLAZINGCORRIDOR] = "Blazing Corridor";
            mapDisplayNames[MAP_STORAGEROOM] = "Storage Room";
            mapDisplayNames[MAP_SPACESHIPENTRANCE] = "Spaceship: Entrance";
            mapDisplayNames[MAP_SPACESHIP01] = "Spaceship: Control Room";
            mapDisplayNames[MAP_SPACESHIP02] = "Spaceship: Closed Entrance";
            mapDisplayNames[MAP_SPACESHIP03] = "Spaceship: Bedroom";
            mapDisplayNames[MAP_SPACESHIPEXIT] = "Spaceship: Exit";
            mapDisplayNames[MAP_MARS01] = "Mars 1";
            mapDisplayNames[MAP_MARS02] = "Mars 2";
            mapDisplayNames[MAP_MARS03] = "Mars 3";
            mapDisplayNames[MAP_MARS04] = "Mars 4";
            mapDisplayNames[MAP_MARS05] = "Mars 5";
            mapDisplayNames[MAP_MARSBASEMENTSTAIRS] = "Mars Basement Stairs";
            mapDisplayNames[MAP_MARSBASEMENT] = "Mars Basement";
            // 175: N/A
            // 176: N/A
            mapDisplayNames[MAP_MALLD] = "The Mall: Top Floor";
            mapDisplayNames[MAP_MALLROOF] = "The Mall: Rooftop";
            mapDisplayNames[MAP_WITCHFLIGHT] = "Witch's Flight";
            
            nexusExitNames = new Dictionary<string, string>();
            nexusExitNames["DoorRoom/01Frog"] = "Green Forest Door";
            nexusExitNames["DoorRoom/02HatAndScarf"] = "Light-Gray Block Door";
            nexusExitNames["DoorRoom/03Umbrella"] = "Red Puddle Door";
            nexusExitNames["DoorRoom/04Knife"] = "Striped Dark Door";
            nexusExitNames["DoorRoom/05YukiOnna"] = "Faded-Purple Snow Door";
            nexusExitNames["DoorRoom/06Hairstyle"] = "Light-Pink Mural Door";
            nexusExitNames["DoorRoom/07Bicycle"] = "Purple-Pink Graffiti Door";
            nexusExitNames["DoorRoom/08EyeballHand"] = "Eyeball Door";
            nexusExitNames["DoorRoom/09Small"] = "Dark-Gray Candle Door";
            nexusExitNames["DoorRoom/10CatEars"] = "Purple-Box Shield-Folk Door";
            nexusExitNames["DoorRoom/11Neon"] = "Neon Lights Door";
            nexusExitNames["DoorRoom/12Lamp"] = "Red-Eye Number Door";
        }
        
        // Initializes various lists of maps that have certain qualities.
        static void initSpecialMapLists()
        {
            mapsWithDreamBeds = new List<int>();
            mapsWithDreamBeds.Add(MAP_DREAMROOM); // Bed ID 1
            mapsWithDreamBeds.Add(MAP_BLOCKWORLD); // Bed ID 2
            mapsWithDreamBeds.Add(MAP_SNOWWORLD); // Bed ID 3
            mapsWithDreamBeds.Add(MAP_CANDLEWORLD); // Bed ID 4
            mapsWithDreamBeds.Add(MAP_ROOMOFBEDS); // Bed ID 5
            
            mapsInFCWorld = new List<int>();
            for (int i = MAP_FCHOUSEOUTSIDE; i <= MAP_FCPIRORIVILLAGE; i++)
            {
                if (i != MAP_FCENTRANCEB) // Entrances are not actually "part of" FC World
                    mapsInFCWorld.Add(i);
            }
            mapsInFCWorld.Add(MAP_FCDUNGEON002GLITCH);
            mapsInFCWorld.Add(MAP_FCOVERWORLD);
            
            mapsWithSnow = new List<int>();
            mapsWithSnow.Add(MAP_SNOWWORLD);
            for (int i = MAP_IGLOO1; i <= MAP_IGLOO7; i++)
                mapsWithSnow.Add(i);
            
            mapsWithRain = new List<int>();
            for (int i = MAP_HIGHWAY; i <= MAP_HIGHWAYD; i++)
                mapsWithRain.Add(i);
            mapsWithRain.Add(MAP_JUKAI2); // Sort of a weird case, rain only persists from Highway, but let's say it's supposed to rain always
            
            mapsWithDarkness = new List<int>();
            mapsWithDarkness.Add(MAP_DARKWORLD);
            mapsWithDarkness.Add(MAP_INSIDESHACKA);
            mapsWithDarkness.Add(MAP_JUKAI3);
            mapsWithDarkness.Add(MAP_FCDUNGEON001);
        }
        
        // Decides a valid random layout, then applies it to data. Returns false if terminated by user.
        static bool pickRandomLayout()
        {
            initMapNames();
            initSpecialMapLists();
            
            md5Hasher = MD5.Create();
            byte[] hashed = md5Hasher.ComputeHash(UNICODE.GetBytes(currentSeed));
            trueSeed = BitConverter.ToInt32(hashed, 0);
            random = new Random(trueSeed);
            
            int layoutsGenerated = 0;
            int layoutThreshold = 1000;
            
            do
            {
                if (layoutsGenerated == layoutThreshold)
                {
                    Console.WriteLine("Still trying (" + layoutsGenerated + " attempts)... Press X key to stop.");
                    layoutThreshold += 1000;
                }
                
                if (layoutsGenerated >= 1000 && Console.KeyAvailable)
                {
                    if (Console.ReadKey(true).Key == ConsoleKey.X)
                    {
                        Console.WriteLine("Process terminated.");
                        return false;
                    }
                }
                
                // Determine fixed random event stuff early on so it can be used when defining warps.
                if (derandomizeEvents)
                {
                    fixedDreamBedID = 1 + random.Next(5);
                    fixedClosetWarpID = 1 + random.Next(8);
                    fixedMiniHellClosetID = 1 + random.Next(4);
                    fixedPurpleVersion = random.Next(8);
                    fixedPurpleClosetSwap = random.Next(4);
                    fixedVillageWarper = 1 + random.Next(6);
                    fixedGlitchEvent = random.Next(2) == 0;
                }
                
                initBaseEffects();
                initBaseEssences();
                initBaseWarps();
                initExitLinks();
                
                randomize();
                layoutsGenerated++;
                
                if (spoilerLogEnabled)
                {
                    if (itemOrderForLog == null)
                        itemOrderForLog = new List<int>();
                    else
                        itemOrderForLog.Clear();
                }
            } while (!validLayout());
            
            if (layoutsGenerated >= 1000)
                Console.WriteLine("Got it! (" + layoutsGenerated + " attempts)");
            else
                debugMessage("Layouts generated: " + layoutsGenerated);
            
            if (makeWarpList)
                clipboardWarpList();
            
            database.applyLayout();
            foreach (int key in maps.Keys)
                maps[key].applyLayout();
            
            return true;
        }
        
        // Randomly comes up with a layout.
        static void randomize()
        {
            if (randomizeEffects)
                shuffleDictionary(ref effects);
            
            if (heckificationType == 2) // Hell, and... Goodbye!
            {
                if (entranceRandoType != 3) // Outside of linked Nightmare mode, just assign outta-Hell exits at nandom, no need to form pairs
                {
                    List<string> outtaHellKeysShuffled = new List<string>(outtaHellKeys);
                    shuffleList(ref outtaHellKeysShuffled);
                    
                    for (int i = 0; i < intoHellKeys.Length; i++)
                    {
                        string locationKey = intoHellKeys[i];
                        string warpKey = outtaHellKeysShuffled[i];
                        exits[locationKey] = exits[warpKey];
                    }
                }
                else // In Nightmare Mode, randomly create three links between the six former entrances into Hell
                {
                    List<int> pairingOrder = new List<int>(new int[] { 0, 1, 2, 3, 4, 5 });
                    shuffleList(ref pairingOrder);
                    
                    for (int i = 0; i < 6; i += 2)
                    {
                        string intoKey1 = intoHellKeys[pairingOrder[i]];
                        string intoKey2 = intoHellKeys[pairingOrder[i + 1]];
                        string outtaKey1 = outtaHellKeys[pairingOrder[i]];
                        string outtaKey2 = outtaHellKeys[pairingOrder[i + 1]];
                        
                        exits[intoKey1] = exits[outtaKey2];
                        exits[intoKey2] = exits[outtaKey1];
                        defineExitLink(intoKey1, intoKey2);
                    }
                }
            }
            
            if (entranceRandoType == 1 || entranceRandoType == 2) // Nexus doors
            {
                // Before shuffling anything, be sure to get the default warps, both from and back to the Nexus, as unchanging Warps.
                Warp[] nexusExits = new Warp[12];
                Warp[] backToNexus = new Warp[12];
                for (int i = 0; i < 12; i++)
                {
                    nexusExits[i] = exits[nexusExitKeys[i]];
                    backToNexus[i] = exits[backToNexusKeys[i]];
                }
                
                List<string> unusedKeyList = new List<string>(exits.Keys);
                List<WarpLink> unusedLinks = new List<WarpLink>(linkedExits);
                
                // Remove all basement maps but 019 (Oni location) as possibilities.
                List<string> keysToRemove = new List<string>();
                List<WarpLink> linksToRemove = new List<WarpLink>();
                foreach (string key in unusedKeyList)
                {
                    if (key.Contains("FCBasement") && !key.Contains("/FCBasement019"))
                        keysToRemove.Add(key);
                }
                foreach (WarpLink link in unusedLinks)
                {
                    if ((link.warp1.Contains("FCBasement") && !link.warp1.Contains("/FCBasement019"))
                     || (link.warp2.Contains("FCBasement") && !link.warp2.Contains("/FCBasement019")))
                        linksToRemove.Add(link);
                }
                foreach (string key in keysToRemove)
                    unusedKeyList.Remove(key);
                foreach (WarpLink link in linksToRemove)
                    unusedLinks.Remove(link);
                
                // Pick warps for each Nexus door.
                for (int i = 0; i < 12; i++)
                {
                    string nexusExitKey = nexusExitKeys[i];
                    Warp nexusExitWarp = nexusExits[i];
                    Warp backToNexusWarp = backToNexus[i];
                    
                    string warpKey = "", otherWarpKey = "";
                    Warp nexusExit = null, returnExit = null, turnaroundExit = null;
                    bool valid;
                    
                    do
                    {
                        if (entranceRandoType == 1) // One-way Nexus doors
                        {
                            // Pick a random exit.
                            warpKey = unusedKeyList[random.Next(unusedKeyList.Count)];
                            unusedKeyList.Remove(warpKey);
                            
                            // Have Nexus door go to that random place.
                            nexusExit = exits[warpKey];
                        }
                        else if (entranceRandoType == 2) // Linked Nexus doors
                        {
                            // Pick a random link.
                            WarpLink warpLink = unusedLinks[random.Next(unusedLinks.Count)];
                            unusedLinks.Remove(warpLink);
                            
                            // Get both warps in the link, with which one is chosen "first" being random.
                            warpKey = warpLink.getEitherWarpKey();
                            otherWarpKey = warpLink.getOtherWarpKey(warpKey);
                            
                            // Have Nexus door use warp from "first" half of the link. Have "second" half of link go back to Nexus.
                            // Then have first half take the second half's former warp, effectively turning you back around.
                            nexusExit = exits[warpKey];
                            returnExit = backToNexusWarp;
                            turnaroundExit = exits[otherWarpKey];
                        }
                        
                        valid = true;
                        if (warpKey.Contains("/RealWorldRoom") // Going back to real world messes with items
                         || warpKey.Contains("/Veranda") // Going back to real world messes with items
                         || warpKey.Contains("/FamiGame1") // Not much of a "map," also it's part of real world
                         || warpKey.Contains("/Image") // Not much of a "map," doesn't even show Face, just gives Essence and wakes you up
                         || warpKey.Contains("/FlyWithBroom") // Seems awkward
                         || warpKey.Contains("/Ending")) // Yeah...
                            valid = false;
                        
                        if (heckificationType == 2 && warpKey.Contains("/RedA")) // Don't choose warps into Hell in Hell and Goodbye mode
                            valid = false;
                    } while (!valid);
                    
                    exits[nexusExitKey] = nexusExit;
                    if (entranceRandoType == 2) // Linked
                    {
                        exits[otherWarpKey] = returnExit;
                        exits[warpKey] = turnaroundExit;
                    }
                }
            }
            else if (entranceRandoType == 3) // Nightmare
            {
                List<string> keyList = new List<string>(exits.Keys);
                List<WarpLink> linkList = new List<WarpLink>(linkedExits);
                
                // Remove keys that shouldn't be shuffled and associated links.
                if (heckificationType == 2)
                {
                    foreach (string key in outtaHellKeys) // After first And Goodbye step, warps located in Hell are unreachable, so remove them
                        keyList.Remove(key);
                }
                
                foreach (string key in noShuffleKeys) // Remove warps set as "don't shuffle" in definition
                    keyList.Remove(key);
                
                List<WarpLink> linksToRemove = new List<WarpLink>();
                
                foreach (WarpLink link in linkList)
                {
                    if (noShuffleKeys.Contains(link.warp1) || noShuffleKeys.Contains(link.warp2))
                        linksToRemove.Add(link);
                }
                foreach (WarpLink link in linksToRemove)
                    linkList.Remove(link);
                
                // Make a list of all two-way warp keys to identify when a warp is one-way.
                List<string> twoWayKeys = new List<string>();
                foreach (WarpLink link in linkList)
                {
                    twoWayKeys.Add(link.warp1);
                    twoWayKeys.Add(link.warp2);
                }
                
                // Create lists of one-way warp keys and their warps.
                List<string> oneWayKeyList = new List<string>();
                List<Warp> oneWayWarpList = new List<Warp>();
                foreach (string key in keyList)
                {
                    if (!twoWayKeys.Contains(key))
                    {
                        oneWayKeyList.Add(key);
                        oneWayWarpList.Add(exits[key]);
                    }
                }
                
                // Shuffle the one-way list and re-assign it to exits list.
                shuffleList(ref oneWayWarpList);
                
                for (int i = 0; i < oneWayKeyList.Count; i++)
                {
                    string key = oneWayKeyList[i];
                    exits[key] = oneWayWarpList[i];
                }
                
                // Keep swapping exits between random pairs of links until all are swapped. Note that a link swapping with itself is possible.
                while (linkList.Count >= 2)
                {
                    int index1 = random.Next(linkList.Count);
                    int index2 = random.Next(linkList.Count);
                    
                    WarpLink link1 = linkList[index1];
                    WarpLink link2 = linkList[index2];
                    if (linkList.Contains(link1))
                        linkList.Remove(link1);
                    if (linkList.Contains(link2))
                        linkList.Remove(link2);
                    
                    if (index1 == index2) // Self-swap
                        continue;
                    
                    // Swap exits in one of two random ways.
                    Warp link1warp1 = exits[link1.warp1];
                    Warp link1warp2 = exits[link1.warp2];
                    Warp link2warp1 = exits[link2.warp1];
                    Warp link2warp2 = exits[link2.warp2];
                    
                    if (random.Next(2) == 0)
                    {
                        exits[link1.warp1] = link2warp2;
                        exits[link1.warp2] = link2warp1;
                        exits[link2.warp1] = link1warp2;
                        exits[link2.warp2] = link1warp1;
                    }
                    else
                    {
                        exits[link1.warp1] = link2warp1;
                        exits[link1.warp2] = link2warp2;
                        exits[link2.warp1] = link1warp1;
                        exits[link2.warp2] = link1warp2;
                    }
                }
            }
            else if (entranceRandoType == 4) // Complete Nightmare
            {
                List<string> keyList = new List<string>(exits.Keys);
                Dictionary<string, Warp> baseExits = new Dictionary<string, Warp>(exits);
                
                // Remove keys that shouldn't be shuffled.
                if (heckificationType == 2)
                {
                    foreach (string key in outtaHellKeys) // After first And Goodbye step, warps located in Hell are unreachable, so remove them
                        keyList.Remove(key);
                }
                
                foreach (string key in noShuffleKeys) // Remove warps set as "don't shuffle" in definition
                    keyList.Remove(key);
                
                // Shuffle the list and re-assign to exits list.
                List<string> shuffledKeyList = new List<string>(keyList);
                shuffleList(ref shuffledKeyList);
                
                for (int i = 0; i < keyList.Count; i++)
                    exits[keyList[i]] = baseExits[shuffledKeyList[i]];
            }
            
            // Pick random effects for each Nexus lock.
            nexusLocks = new int[12];
            if (nexusLockCount > 0)
            {
                List<int> effectList = new List<int>(allEffects);
                int lockIndex;
                for (int i = 0; i < nexusLockCount; i++)
                {
                    do
                    {
                        lockIndex = random.Next(12);
                    } while (nexusLocks[lockIndex] != 0);
                    int effectIndex = random.Next(effectList.Count);
                    nexusLocks[lockIndex] = effectList[effectIndex];
                    effectList.RemoveAt(effectIndex);
                }
            }
            
            // After shuffling, sort location keys into lists by map for quicker access.
            sortLocationKeysByMap(ref effectLocations, ref effectKeysForMap);
            sortLocationKeysByMap(ref essenceLocations, ref essenceKeysForMap);
            sortLocationKeysByMap(ref warpLocations, ref warpKeysForMap);
            
            // Handle early placement of Bicycle/Eyeball-Hand.
            // Since map key lists are needed to search for early Effect locations, it has to wait until after creating said lists.
            if (randomizeEffects && (bicyclePlacement == 1 || eyeballPlacement == 1))
            {
                swapInEarlyEffects();
                sortLocationKeysByMap(ref effectLocations, ref effectKeysForMap); // After swapping in early Effects, refresh Effect reference list
            }
        }
        
        // If "early Effects" options are on, find early location(s) to swap them into.
        static void swapInEarlyEffects()
        {
            findItemLocationsByDepth();
            
            bool bicyclePlaced = false, eyeballPlaced = false;
            if (bicyclePlacement != 1) // Consider "already placed" if it doesn't need to be swapped around
                bicyclePlaced = true;
            if (eyeballPlacement != 1)
                eyeballPlaced = true;
            
            // Search at the lowest depths available for possible locations.
            for (int depth = 0; depth < 200; depth++)
            {
                if (bicyclePlaced && eyeballPlaced)
                    return;
                
                if (effectLocationsAtDepth.ContainsKey(depth))
                {
                    while (effectLocationsAtDepth[depth].Count > 0) // Keep searching at this depth as long as there's locations in it
                    {
                        int index = random.Next(effectLocationsAtDepth[depth].Count);
                        string earlyLocationKey = effectLocationsAtDepth[depth][index];
                        
                        // Once a location is picked to swap into, remove that location from all depths so it isn't chosen again.
                        for (int i = depth; i < 200; i++)
                            if (effectLocationsAtDepth.ContainsKey(i))
                                effectLocationsAtDepth[i].Remove(earlyLocationKey);
                        
                        int formerEffect = effects[earlyLocationKey];
                        
                        if (!bicyclePlaced)
                        {
                            string bicycleLocationKey = "";
                            foreach (string key in effectLocations.Keys)
                            {
                                if (effects[key] == EF_BICYCLE)
                                {
                                    bicycleLocationKey = key;
                                    break;
                                }
                            }
                            
                            effects[earlyLocationKey] = EF_BICYCLE;
                            effects[bicycleLocationKey] = formerEffect;
                            bicyclePlaced = true;
                            
                            if (bicyclePlaced && eyeballPlaced)
                                return;
                            else
                                continue; // Restart while loop to pick new location, lest Eyeball-Hand use the same one and swap Bicycle out into wherever
                        }
                        
                        if (!eyeballPlaced)
                        {
                            string eyeballLocationKey = "";
                            foreach (string key in effectLocations.Keys)
                            {
                                if (effects[key] == EF_EYEBALLHAND)
                                {
                                    eyeballLocationKey = key;
                                    break;
                                }
                            }
                            
                            effects[earlyLocationKey] = EF_EYEBALLHAND;
                            effects[eyeballLocationKey] = formerEffect;
                            eyeballPlaced = true;
                            
                            if (bicyclePlaced && eyeballPlaced)
                                return;
                        }
                    }
                }
            }
        }
        
        // Creates depth-based index of Effect and Essence locations.
        static void findItemLocationsByDepth()
        {
            simulationEffects = new List<int>();
            pendingEffects = new List<int>();
            simulationEssences = new List<int>();
            pendingEssences = new List<int>();
            simulationMapsChecked = new List<string>();
            
            effectLocationsAtDepth = new Dictionary<int, List<string>>();
            effectLocationMapChains = new Dictionary<int, List<string>>();
            essenceLocationsAtDepth = new Dictionary<int, List<string>>();
            essenceLocationMapChains = new Dictionary<int, List<string>>();
            
            while (simulationEffects.Count < 24 && simulationEssences.Count < 8)
            {
                simulationMapsChecked.Clear();
                simulationProgressMade = false;
                
                findingItemLocationsByDepth = true;
                
                maps[MAP_REALWORLDROOM].searchForProgress();
                
                // Add Effects and Essences gained in this search.
                foreach (int effect in pendingEffects)
                    simulationEffects.Add(effect);
                foreach (int essence in pendingEssences)
                    simulationEssences.Add(essence);
                pendingEffects.Clear();
                pendingEssences.Clear();
                
                findingItemLocationsByDepth = false;
                
                if (!simulationProgressMade) // No further progress possible, so stop here
                    break;
            }
        }
        
        // Goes through a dictionary of locations to make a list of applicable keys for each map.
        static void sortLocationKeysByMap(ref Dictionary<string, CommandLocation> locations, ref Dictionary<int, List<string>> keysForMap)
        {
            keysForMap = new Dictionary<int, List<string>>();
            foreach (string key in locations.Keys)
            {
                if (locations[key] is MapLocation)
                {
                    MapLocation location = locations[key] as MapLocation;
                    int map = location.mapID;
                    if (!keysForMap.ContainsKey(map))
                        keysForMap[map] = new List<string>();
                    keysForMap[map].Add(key);
                }
                else if (locations[key] is CombinedLocation)
                {
                    CombinedLocation combo = locations[key] as CombinedLocation;
                    foreach (MapLocation location in combo.locations)
                    {
                        int map = location.mapID;
                        if (!keysForMap.ContainsKey(map))
                            keysForMap[map] = new List<string>();
                        keysForMap[map].Add(key);
                    }
                }
            }
        }
        
        // Simulates a playthrough and returns if the game layout is valid.
        static bool validLayout()
        {
            simulationEffects = new List<int>();
            pendingEffects = new List<int>();
            simulationEssences = new List<int>();
            pendingEssences = new List<int>();
            simulationMapsChecked = new List<string>();
            
            if (bicyclePlacement == 2) // Start with it
                simulationEffects.Add(EF_BICYCLE);
            if (eyeballPlacement == 2) // Start with it
                simulationEffects.Add(EF_EYEBALLHAND);
            
            while (simulationEffects.Count < 24 || simulationEssences.Count < 8)
            {
                simulationMapsChecked.Clear();
                simulationProgressMade = false;
                
                maps[MAP_REALWORLDROOM].searchForProgress();
                
                // Add Effects and Essences gained in this search.
                foreach (int effect in pendingEffects)
                    simulationEffects.Add(effect);
                foreach (int essence in pendingEssences)
                    simulationEssences.Add(essence);
                pendingEffects.Clear();
                pendingEssences.Clear();
                
                // If no spoiler log, only requirement is to reach the minimums. Spoiler log should keep going as far as possible, though.
                if (simulationEffects.Count >= minimumEffects && simulationEssences.Count >= minimumEssences && !spoilerLogEnabled)
                    return true;
                
                if (!simulationProgressMade) // No further progress possible, so check if minimums have been reached
                {
                    if (simulationEffects.Count >= minimumEffects && simulationEssences.Count >= minimumEssences)
                        return true;
                    else
                    {
                        if (showMissingItems)
                            printUnobtainedItems();
                        return false;
                    }
                }
            }
            
            return true; // Achieved 100%
        }
        
        // Debug function to list what items were not obtained in the previous simulation.
        static void printUnobtainedItems()
        {
            List<int> effectList = new List<int>(allEffects);
            List<int> essenceList = new List<int>(allEssences);
            
            foreach (int effect in simulationEffects)
                effectList.Remove(effect);
            foreach (int essence in simulationEssences)
                essenceList.Remove(essence);
            
            Console.WriteLine("Giving up with " + simulationEffects.Count + " Effects, " + simulationEssences.Count + " Essences");
            
            foreach (int effect in effectList)
                Console.WriteLine(effectProperName[effect]);
            foreach (int essence in essenceList)
                Console.WriteLine(essenceProperName[essence]);
        }
        
        // Debug function to put a list of warps on the clipboard.
        static void clipboardWarpList()
        {
            StringWriter writer = new StringWriter(new StringBuilder());
            
            foreach (string key in exits.Keys)
            {
                Warp warp = exits[key];
                string locationStr = "";
                
                if (warpLocations[key] is MapLocation)
                {
                    MapLocation location = warpLocations[key] as MapLocation;
                    string section = location.section != ""? " (" + location.section + ")" : "";
                    locationStr = mapDisplayNames[location.mapID] + section;
                }
                else if (warpLocations[key] is CommonLocation)
                {
                    CommonLocation location = warpLocations[key] as CommonLocation;
                    locationStr = "Common Event " + location.eventID;
                }
                else if (warpLocations[key] is CombinedLocation)
                {
                    MapLocation location = (warpLocations[key] as CombinedLocation).locations[0] as MapLocation;
                    string section = location.section != ""? " (" + location.section + ")" : "";
                    locationStr = mapDisplayNames[location.mapID] + section;
                }
                
                string warpSection = warp.section != ""? " (" + warp.section + ")" : "";
                
                writer.WriteLine(key);
                writer.WriteLine(locationStr + " -> " + mapDisplayNames[warp.mapID] + warpSection + " (" + warp.x + "," + warp.y + ")");
                writer.WriteLine();
            }
            
            Clipboard.SetText(writer.ToString());
            Console.WriteLine("Exit list put on clipboard.");
        }
        
        // Saves the randomized data in a new game folder.
        static void saveRandomGame()
        {
            string newRoot = "Rem9-v" + versionID + (userSettings["IncludeSeedInFolderName"] == 1? "-" + currentSeed : "");
            Directory.CreateDirectory(newRoot);
            
            // Copy launchers and original readme, and create extra text files.
            string baseRoot = Directory.GetParent(gamePath).FullName;
            
            copyBetweenFolders(baseRoot, newRoot, "readme.txt");
            copyBetweenFolders(baseRoot, newRoot, "StartWindowed.exe");
            copyBetweenFolders(baseRoot, newRoot, "StartFullscreen.exe");
            
            generateSeedInfoFile(newRoot + "\\SeedInfo.txt");
            
            if (spoilerLogEnabled)
                writeSpoilerLog(newRoot + "\\SpoilerLog.txt");
            else if (File.Exists(newRoot + "\\SpoilerLog.txt"))
                File.Delete(newRoot + "\\SpoilerLog.txt");
            
            // Copy RPG_RT.exe and .ini.
            string newPath = newRoot + "\\Data";
            Directory.CreateDirectory(newPath);
            
            copyBetweenFolders(gamePath, newPath, "RPG_RT.exe");
            copyBetweenFolders(gamePath, newPath, "RPG_RT.ini");
            
            // Copy resource directories.
            IEnumerable<string> folderList = Directory.EnumerateDirectories(gamePath);
            foreach (string folder in folderList)
            {
                string folderName = Path.GetFileName(folder);
                if (!folderName.Equals("Title"))
                    copyDirectory(folder, newPath + "\\" + folderName);
            }
            
            // Remove save files from previous seeds.
            IEnumerable<string> saveList = Directory.EnumerateFiles(newPath, "Save*.lsd");
            foreach (string saveFile in saveList)
                File.Delete(saveFile);
            
            // Save randomized data files.
            mapTree.writeFile(newPath + "\\RPG_RT.lmt");
            database.writeFile(newPath + "\\RPG_RT.ldb");
            foreach (int key in maps.Keys)
                maps[key].writeFile(newPath + "\\Map" + key.ToString("D4") + ".lmu");
            
            writeHashToTitleImage(newPath);
            
            Console.WriteLine("Game generated in " + newRoot + "!"
                + (userSettings["IncludeSeedInFolderName"] != 1? " (Seed: " + currentSeed + ")" : ""));
        }
        
        // Finds shortest paths to each item and writes the spoiler log to file.
        static void writeSpoilerLog(string filepath)
        {
            findItemLocationsByDepth();
            
            spoilerLog = new StringWriter(new StringBuilder());
            bool notEmpty = false;
            
            if (nexusLockCount > 0 || entranceRandoType > 0) // Nexus info only relevant if Nexus locks or any form of entrance randomization are enabled
            {
                if (nexusLockCount > 0)
                    spoilerLog.WriteLine("< Nexus Locks and Destinations >");
                else
                    spoilerLog.WriteLine("< Nexus Door Destinations >");
                spoilerLog.WriteLine();
                
                for (int i = 0; i < 12; i++)
                {
                    string key = nexusExitKeys[i];
                    Warp exit = exits[key];
                    string lockStr = nexusLocks[i] != 0? " (requires " + effectProperName[nexusLocks[i]] + ")" : "";
                    spoilerLog.WriteLine(nexusExitNames[key] + lockStr + ":");
                    spoilerLog.WriteLine(mapDisplayNames[exit.mapID] + (exit.section != ""? " (" + exit.section + ")" : ""));
                    spoilerLog.WriteLine();
                }
                notEmpty = true;
            }
            
            if (derandomizeEvents) // If Staircase of Hands bed is predetermined, say which one it is
            {
                spoilerLog.WriteLine("< Other Information >");
                spoilerLog.WriteLine();
                
                spoilerLog.WriteLine(mapDisplayNames[mapsWithDreamBeds[fixedDreamBedID - 1]] + " has guaranteed bed to Staircase of Hands");
                spoilerLog.WriteLine();
                notEmpty = true;
            }
            
            // Print out item acquiring paths, in the order they were acquired during the simulation.
            int tierNumber = 0;
            for (int i = 0; i < itemOrderForLog.Count; i++)
            {
                int itemID = itemOrderForLog[i];
                
                if (itemID == 0) // Loop start marker
                {
                    if (notEmpty) // Don't include break if nothing was written yet
                    {
                        spoilerLog.WriteLine("----");
                        spoilerLog.WriteLine();
                        spoilerLog.WriteLine("< Tier " + tierNumber + " Item Locations (" + (tierNumber == 0? "accessible from start" : "one or more items from previous tier required") + ") >");
                        spoilerLog.WriteLine();
                        tierNumber++;
                    }
                }
                else
                {
                    if (itemID >= EF_FROG && itemID <= EF_STOPLIGHT) // Item is Effect
                        writeShortestPathToItem(itemID, ref effects, ref effectLocations, ref effectLocationsAtDepth, ref effectLocationMapChains, ref effectProperName);
                    else // Item is Essence
                        writeShortestPathToItem(itemID, ref essences, ref essenceLocations, ref essenceLocationsAtDepth, ref essenceLocationMapChains, ref essenceProperName);
                    notEmpty = true;
                }
            }
            
            writeToNewFile(filepath, spoilerLog.ToString());
        }
        
        // After creating index of locations by depth, searches for lowest-depth path to given item and writes it to spoiler log.
        static void writeShortestPathToItem(int itemID, ref Dictionary<string, int> items, ref Dictionary<string, CommandLocation> locations,
            ref Dictionary<int, List<string>> locationsAtDepth, ref Dictionary<int, List<string>> locationMapChains, ref Dictionary<int, string> properName)
        {
            for (int depth = 0; depth < 200; depth++)
            {
                if (locationsAtDepth.ContainsKey(depth))
                {
                    for (int i = 0; i < locationsAtDepth[depth].Count; i++)
                    {
                        string locationKey = locationsAtDepth[depth][i];
                        if (items[locationKey] == itemID)
                        {
                            string mapChain = locationMapChains[depth][i];
                            mapChain = mapChain.Replace("Real-World Room -> Dream Veranda A -> ", "");
                            mapChain = mapChain.Replace("Real-World Room -> Dream Veranda B -> ", "");
                            
                            MapLocation location = null;
                            if (locations[locationKey] is MapLocation)
                                location = locations[locationKey] as MapLocation;
                            else if (locations[locationKey] is CombinedLocation)
                                location = (locations[locationKey] as CombinedLocation).locations[0] as MapLocation;
                            
                            string locationName = mapDisplayNames[location.mapID] + (location.section != ""? " (" + location.section + ")" : "");
                            string itemName = properName[itemID];
                            spoilerLog.WriteLine(locationName + " - " + itemName);
                            spoilerLog.WriteLine(mapChain);
                            spoilerLog.WriteLine();
                            return;
                        }
                    }
                }
            }
        }
        
        // Creates a file in randomized game folder detailing the seed, version, and settings used.
        static void generateSeedInfoFile(string filename)
        {
            StringWriter writer = new StringWriter(new StringBuilder());
            
            writer.WriteLine("Generated with REM-9 ver. " + versionStr);
            writer.WriteLine("Hash Value: " + getHashString());
            writer.WriteLine("Random Seed Used: " + currentSeed);
            writer.WriteLine();
            
            writer.WriteLine("--- Difficulty Settings ---");
            writer.WriteLine("Randomize Effects: " + onOff(randomizeEffects));
            writer.WriteLine("Nexus Locks: " + nexusLockCount);
            writer.WriteLine("Entrance Randomization: " + entranceRandoTypes[entranceRandoType]);
            writer.WriteLine("Bicycle Placement: " + itemPlacements[bicyclePlacement]);
            writer.WriteLine("Eyeball-Hand Placement: " + itemPlacements[eyeballPlacement]);
            writer.WriteLine("Nexus Lock Hints: " + onOff(nexusLockHints));
            writer.WriteLine("Generate Spoiler Log: " + onOff(spoilerLogEnabled));
            writer.WriteLine();
            
            writer.WriteLine("--- Goal Settings ---");
            writer.WriteLine("Effects Goal: " + effectsGoal);
            writer.WriteLine("Essences Goal: " + essencesGoal);
            writer.WriteLine("Minimum Reachable Effects: " + minimumEffects);
            writer.WriteLine("Minimum Reachable Essences: " + minimumEssences);
            writer.WriteLine();
            
            writer.WriteLine("--- Game Settings ---");
            writer.WriteLine("Derandomize Events: " + onOff(derandomizeEvents));
            writer.WriteLine("Fix Chair Glitch: " + onOff(fixChairGlitch));
            writer.WriteLine("Walk Speed: " + walkSpeed);
            writer.WriteLine("Bike Speed: " + bikeSpeed);
            writer.WriteLine("Walk Sound: " + walkSounds[walkSound]);
            writer.WriteLine("Heckification Type: " + heckificationTypes[heckificationType]);
            writer.WriteLine("Easy Teleporter Maze: " + onOff(easyTeleporterMaze));
            writer.WriteLine("Darkness Level: " + darknessLevels[darknessLevel]);
            writer.WriteLine("Givers Reflect Effect: " + onOff(giversReflectEffect));
            
            writeToNewFile(filename, writer.ToString());
        }
        
        // Loads title graphic and writes a hash string onto it.
        static void writeHashToTitleImage(string newPath)
        {
            try
            {
                Bitmap titleBitmap = new Bitmap(gamePath + "\\Title\\Title.bmp");
                Bitmap fcNumbers = new Bitmap(gamePath + "\\Title\\FCNumbers.bmp");
                
                // Find the index for the desired gray in the title image's palette.
                byte grayIndex = 0;
                for (int i = 0; i < titleBitmap.Palette.Entries.Length; i++)
                {
                    Color paletteColor = titleBitmap.Palette.Entries[i];
                    if (paletteColor.R == 63 && paletteColor.G == 63 && paletteColor.B == 63)
                    {
                        grayIndex = (byte)i;
                        break;
                    }
                }
                
                // Get pixel data for the numbers bitmap.
                BitmapData numbersData = fcNumbers.LockBits(new Rectangle(0, 0, fcNumbers.Width, fcNumbers.Height),
                                                            ImageLockMode.ReadOnly, fcNumbers.PixelFormat);
                IntPtr pointer = numbersData.Scan0;
                int bytes = Math.Abs(numbersData.Stride) * fcNumbers.Height;
                byte[] pixelData = new byte[bytes];
                Marshal.Copy(pointer, pixelData, 0, bytes);
                
                // Using numbers image, make a bool array of what pixels should be "filled" to create each number.
                bool[][][] numberPixels = new bool[10][][]; // [number][x][y]
                for (int i = 0; i < 10; i++)
                {
                    numberPixels[i] = new bool[8][];
                    for (int x = 0; x < 8; x++)
                    {
                        numberPixels[i][x] = new bool[8];
                        for (int y = 0; y < 8; y++)
                            numberPixels[i][x][y] = pixelData[(i * 8) + x + (y * numbersData.Width)] == 1; // If pixel index is 1 (gray), it's a filled pixel
                    }
                }
                
                Marshal.Copy(pixelData, 0, pointer, bytes);
                fcNumbers.UnlockBits(numbersData);
                
                // Get the hash string to be printed.
                string hashStr = getHashString();
                
                // Get pixel data for title bitmap.
                BitmapData titleData = titleBitmap.LockBits(new Rectangle(0, 0, titleBitmap.Width, titleBitmap.Height),
                                                            ImageLockMode.WriteOnly, titleBitmap.PixelFormat);
                pointer = titleData.Scan0;
                bytes = Math.Abs(titleData.Stride) * titleBitmap.Height;
                pixelData = new byte[bytes];
                Marshal.Copy(pointer, pixelData, 0, bytes);
                
                // Paste the "filled pixels" for each number into the title bitmap data array, setting pixels to the gray palette index determined earlier.
                int leftX = (int)(160 - ((hashStr.Length / 2f) * 8)), topY = 216;
                for (int i = 0; i < hashStr.Length; i++)
                {
                    int num = int.Parse(hashStr[i].ToString());
                
                    for (int x = 0; x < 8; x++)
                    {
                        for (int y = 0; y < 8; y++)
                        {
                            if (numberPixels[num][x][y])
                                pixelData[leftX + (i * 8) + x + ((topY + y) * titleData.Stride)] = grayIndex;
                        }
                    }
                }
                
                Marshal.Copy(pixelData, 0, pointer, bytes);
                titleBitmap.UnlockBits(titleData);
                
                // Save the finished bitmap.
                Directory.CreateDirectory(newPath + "\\Title");
                titleBitmap.Save(newPath + "\\Title\\Title.bmp", ImageFormat.Bmp);
            }
            catch (Exception e)
            {
                M.debugMessage(e.StackTrace);
                M.debugMessage(e.Message);
                Console.WriteLine("Warning: Could not write hash value to title screen. Using default title.");
                
                Directory.CreateDirectory(newPath + "\\Title");
                copyBetweenFolders(gamePath + "\\Title", newPath + "\\Title", "Title.bmp");
            }
        }
        
        // Creates a hash string based on seed, version, and settings.
        static string getHashString()
        {
            List<byte> settings = new List<byte>();
            settings.AddRange(BitConverter.GetBytes(trueSeed));
            settings.AddRange(UNICODE.GetBytes(versionStr));
            
            settings.Add((byte)(randomizeEffects? 1 : 0));
            settings.Add((byte)nexusLockCount);
            settings.Add((byte)entranceRandoType);
            settings.Add((byte)bicyclePlacement);
            settings.Add((byte)eyeballPlacement);
            settings.Add((byte)(nexusLockHints? 1 : 0));
            settings.Add((byte)(spoilerLogEnabled? 1 : 0));
            
            settings.Add((byte)effectsGoal);
            settings.Add((byte)essencesGoal);
            settings.Add((byte)minimumEffects);
            settings.Add((byte)minimumEssences);
            
            settings.Add((byte)(derandomizeEvents? 1 : 0));
            settings.Add((byte)(fixChairGlitch? 1 : 0));
            settings.Add((byte)walkSpeed);
            settings.Add((byte)bikeSpeed);
            settings.Add((byte)walkSound);
            settings.Add((byte)heckificationType);
            settings.Add((byte)(easyTeleporterMaze? 1 : 0));
            settings.Add((byte)darknessLevel);
            settings.Add((byte)(giversReflectEffect? 1 : 0));
            
            byte[] hashed = md5Hasher.ComputeHash(settings.ToArray());
            return Math.Abs(BitConverter.ToInt32(hashed, 0)).ToString();
        }
        
        // Loads user settings file. Option settings are only loaded once on startup, while others can be updated mid-program.
        static void loadUserSettings(bool loadAllSettings = false)
        {
            if (userSettings == null)
                userSettings = new Dictionary<string, int>();
            if (userSettingsStr == null)
                userSettingsStr = new Dictionary<string, string>();
            
            // Default values for non-program-option user preferences.
            userSettings["IncludeSeedInFolderName"] = 0;
            
            // Non-program-option user preferences should be reloaded frequently.
            List<string> alwaysReload = new List<string>();
            alwaysReload.Add("IncludeSeedInFolderName");
            
            // Default values for program options; only get into these on initial load.
            if (loadAllSettings)
            {
                userSettings["RandomizeEffects"] = 1;
                userSettings["NexusLockCount"] = 8;
                userSettings["EntranceRandomization"] = 1;
                userSettings["BicyclePlacement"] = 0;
                userSettings["EyeballPlacement"] = 0;
                userSettings["NexusLockHints"] = 0;
                userSettings["SpoilerLogEnabled"] = 0;
                
                userSettings["EffectsGoal"] = 24;
                userSettings["EssencesGoal"] = 0;
                userSettings["MinimumEffects"] = 24;
                userSettings["MinimumEssences"] = 0;
                
                userSettings["DerandomizeEvents"] = 1;
                userSettings["FixChairGlitch"] = 1;
                userSettings["WalkSpeed"] = 4;
                userSettings["BikeSpeed"] = 5;
                userSettings["WalkSound"] = 0;
                userSettings["HeckificationType"] = 1;
                userSettings["EasyTeleporterMaze"] = 1;
                userSettings["DarknessLevel"] = 1;
                userSettings["GiversReflectEffect"] = 0;
            }
            
            // Load settings from file, but after initial load, skip over non-reloaded ones.
            if (File.Exists("UserSettings.txt"))
            {
                using (StreamReader input = new StreamReader("UserSettings.txt"))
                {
                    string line;
                    while ((line = input.ReadLine()) != null)
                    {
                        if (line.StartsWith("@"))
                        {
                            int settingStart = line.IndexOf("@") + 1;
                            int equals = line.IndexOf("=");
                            string settingName = line.Substring(settingStart, equals - 1).Trim();
                            string settingStr = line.Substring(equals + 1).Trim();
                            
                            if (!loadAllSettings && !alwaysReload.Contains(settingName))
                                continue;
                            
                            if (userSettingsStr.ContainsKey(settingName)) // String
                                userSettingsStr[settingName] = settingStr;
                            else // Number
                            {
                                int setting;
                                if (int.TryParse(settingStr, out setting))
                                    userSettings[settingName] = setting;
                            }
                        }
                    }
                }
            }
            
            // Validate non-program-option user preferences.
            validateUserSetting("IncludeSeedInFolderName", 0, 1, 0);
            
            // Validate program options.
            validateUserSetting("RandomizeEffects", 0, 1, 1);
            validateUserSetting("NexusLockCount", 0, 11, 8);
            validateUserSetting("EntranceRandomization", 0, 4, 1);
            validateUserSetting("BicyclePlacement", 0, 2, 0);
            validateUserSetting("EyeballPlacement", 0, 2, 0);
            validateUserSetting("NexusLockHints", 0, 1, 0);
            validateUserSetting("SpoilerLogEnabled", 0, 1, 0);
            
            validateUserSetting("EffectsGoal", 1, 24, 24);
            validateUserSetting("EssencesGoal", 0, 8, 0);
            validateUserSetting("MinimumEffects", 1, 24, 24);
            validateUserSetting("MinimumEssences", 0, 8, 0);
            
            validateUserSetting("DerandomizeEvents", 0, 1, 1);
            validateUserSetting("FixChairGlitch", 0, 1, 1);
            validateUserSetting("WalkSpeed", 1, 6, 4);
            validateUserSetting("BikeSpeed", 1, 6, 5);
            validateUserSetting("WalkSound", 0, 2, 0);
            validateUserSetting("HeckificationType", 0, 2, 1);
            validateUserSetting("EasyTeleporterMaze", 0, 1, 1);
            validateUserSetting("DarknessLevel", 0, 2, 1);
            validateUserSetting("GiversReflectEffect", 0, 1, 0);
            
            // On first load, set program options to the specified defaults.
            if (loadAllSettings)
            {
                randomizeEffects = userSettings["RandomizeEffects"] == 1;
                nexusLockCount = userSettings["NexusLockCount"];
                entranceRandoType = userSettings["EntranceRandomization"];
                bicyclePlacement = userSettings["BicyclePlacement"];
                eyeballPlacement = userSettings["EyeballPlacement"];
                nexusLockHints = userSettings["NexusLockHints"] == 1;
                spoilerLogEnabled = userSettings["SpoilerLogEnabled"] == 1;
                
                effectsGoal = userSettings["EffectsGoal"];
                essencesGoal = userSettings["EssencesGoal"];
                minimumEffects = userSettings["MinimumEffects"];
                minimumEssences = userSettings["MinimumEssences"];
                if (minimumEffects < effectsGoal)
                    minimumEffects = effectsGoal;
                if (minimumEssences < essencesGoal)
                    minimumEssences = essencesGoal;
                
                derandomizeEvents = userSettings["DerandomizeEvents"] == 1;
                fixChairGlitch = userSettings["FixChairGlitch"] == 1;
                walkSpeed = userSettings["WalkSpeed"];
                bikeSpeed = userSettings["BikeSpeed"];
                walkSound = userSettings["WalkSound"];
                heckificationType = userSettings["HeckificationType"];
                easyTeleporterMaze = userSettings["EasyTeleporterMaze"] == 1;
                darknessLevel = userSettings["DarknessLevel"];
                giversReflectEffect = userSettings["GiversReflectEffect"] == 1;
            }
        }
        static void validateUserSetting(string settingName, int min, int max, int def)
        {
            if (userSettings[settingName] < min || userSettings[settingName] > max)
                userSettings[settingName] = def;
        }
        
        // Checks data for warp commands and generates a list of definitions.
        static void compileWarps()
        {
            warpDefinitionList = new StringWriter(new StringBuilder());
            
            loadBaseGame();
            
            database.getWarps();
            foreach (int mapID in maps.Keys)
                maps[mapID].getWarps();
            
            Clipboard.SetText(warpDefinitionList.ToString());
            
            debugMessage("Warp list put on clipboard.");
        }
        
        // Checks warp definitions for ones that might link to each other (based on map IDs) and generates a list of link definitions.
        static void compileExitLinks()
        {
            fixedMiniHellClosetID = 1; // Just go with a temporary one
            initBaseWarps();
            
            StringWriter writer = new StringWriter(new StringBuilder());
            
            List<int> outgoingMaps = new List<int>();
            List<string> outgoingName = new List<string>();
            int lastMapWritten = 0;
            
            for (int i = 1; i <= 179; i++) // Check all maps in order
            {
                outgoingMaps.Clear();
                outgoingName.Clear();
                
                foreach (string key in warpLocations.Keys)
                {
                    if (warpLocations[key] is MapLocation)
                    {
                        MapLocation location = warpLocations[key] as MapLocation;
                        if (location.mapID == i) // For every warp located in the current map...
                        {
                            outgoingMaps.Add(exits[key].mapID); // Add the map it goes to to the list
                            outgoingName.Add(key); // As well as the name of the warp
                        }
                    }
                }
                
                foreach (string key in warpLocations.Keys)
                {
                    if (warpLocations[key] is MapLocation)
                    {
                        MapLocation location = warpLocations[key] as MapLocation;
                        if (outgoingMaps.Contains(location.mapID)) // For every warp located in one of the outgoing maps...
                        {
                            if (exits[key].mapID == i) // If the warp goes back to the first map...
                            {
                                if (i < location.mapID) // And it's in incrementing order, so it's not just the reverse of an earlier link...
                                {
                                    int outgoingIndex = outgoingMaps.IndexOf(location.mapID);
                                    string warpName = outgoingName[outgoingIndex];
                                    if (lastMapWritten != i)
                                    {
                                        writer.WriteLine();
                                        lastMapWritten = i;
                                    }
                                    writer.WriteLine("defineExitLink(\"" + warpName + "\", \"" + key + "\");");
                                }
                            }
                        }
                    }
                }
            }
            
            Clipboard.SetText(writer.ToString());
        }
        
        /* BYTE-READING */
        
        // Reads a single byte.
        public static byte readByte(FileStream f)
        {
            return (byte)f.ReadByte();
        }
        
        // Reads and returns the value of a int16 (two bytes).
        public static int readTwoBytes(FileStream f)
        {
            int total = readByte(f);
            total += readByte(f) * 256;
            return total;
        }
        
        // Reads and returns the value of a int32 (four bytes).
        public static long readFourBytes(FileStream f)
        {
            int total = readByte(f);
            total += readByte(f) * 256;
            total += readByte(f) * 65536;
            total += readByte(f) * 16777216;
            return total;
        }
        
        // Reads and returns the value of a "multibyte," used for values over 127 (0x7f).
        public static int readMultibyte(FileStream f, int sum = 0)
        {
            byte b = readByte(f);
            if (b < 128)
                return b + sum;
            else
                return readMultibyte(f, (b - 128 + sum) * 128);
        }
        
        // Reads a byte length (always 1), then a 0 or 1.
        public static bool readLengthBool(FileStream f)
        {
            byteCheck(f, 0x01);
            return (readByte(f) == 1);
        }
        
        // Reads a byte length, then a number of bytes dictated by the length.
        public static int readLengthBytes(FileStream f)
        {
            int length = readMultibyte(f);
            if (length == 1)
                return readByte(f);
            else if (length == 2)
                return readTwoBytes(f);
            else
                return readMultibyte(f);
        }
        
        // Reads a byte length, then a multibyte.
        public static int readLengthMultibyte(FileStream f)
        {
            readMultibyte(f);
            return readMultibyte(f);
        }
        
        // Reads a byte length (always 8), then an eight-byte double.
        public static double readLengthDouble(FileStream f)
        {
            byteCheck(f, 0x08);
            BinaryReader reader = new BinaryReader(f);
            return reader.ReadDouble();
        }
        
        // Reads a byte length, then a byte or two, which is translated into bools based on the bits.
        public static bool[] readLengthFlags(FileStream f)
        {
            int length = readByte(f);
            int flagCount = length * 8;
            int bits = length == 1? readByte(f) : readTwoBytes(f);
            
            bool[] flags = new bool[flagCount];
            for (int i = 0; i < flagCount; i++)
                flags[i] = (bits & (int)Math.Pow(2, i)) != 0;
            
            return flags;
        }
        
        // Reads a byte length, then an array of bytes.
        public static int[] readByteArray(FileStream f)
        {
            int length = readMultibyte(f);
            int[] array = new int[length];
            for (int i = 0; i < length; i++)
                array[i] = readByte(f);
            return array;
        }
        
        // Reads a byte length, then an array of two-byte integers.
        public static int[] readTwoByteArray(FileStream f)
        {
            int length = readMultibyte(f);
            if (length < 2) // Length can be 0 or 1 when "empty"
            {
                int[] array = new int[length];
                for (int i = 0; i < length; i++)
                    array[i] = readByte(f);
                return array;
            }
            else
            {
                int[] array = new int[length / 2];
                for (int i = 0; i < length / 2; i++)
                    array[i] = readTwoBytes(f);
                return array;
            }
        }
        
        // Reads a byte length, then an array of four-byte integers.
        public static long[] readFourByteArray(FileStream f)
        {
            int length = readMultibyte(f);
            long[] array = new long[length / 4];
            for (int i = 0; i < length / 4; i++)
                array[i] = readFourBytes(f);
            return array;
        }
        
        // Reads a count, then an array of multibytes.
        public static int[] readMultibyteArray(FileStream f)
        {
            int count = readMultibyte(f);
            int[] array = new int[count];
            for (int i = 0; i < count; i++)
                array[i] = readMultibyte(f);
            return array;
        }
        
        // Reads a byte length, then an 2D array of two-byte integers. (Must provide dimension sizes, but size2 can be guessed from length and size1.)
        public static int[][] readTwoByteArray2D(FileStream f, int size1, int size2 = -1)
        {
            int length = readMultibyte(f);
            if (size2 == -1)
                size2 = (length / size1) / 2;
            
            int[][] array = new int[size1][];
            for (int i = 0; i < size1; i++)
            {
                array[i] = new int[size2];
                for (int j = 0; j < size2; j++)
                    array[i][j] = readTwoBytes(f);
            }
            return array;
        }
        
        // Reads a byte length, then an array of bools (0s or 1s).
        public static bool[] readBoolArray(FileStream f)
        {
            int length = readMultibyte(f);
            bool[] array = new bool[length];
            for (int i = 0; i < length; i++)
                array[i] = readByte(f) == 1;
            return array;
        }
        
        // Reads a byte length, then the string of that length following it.
        public static string readString(FileStream f, int strType)
        {
            int length = readMultibyte(f);
            
            byte[] bytes = new byte[length];
            for (int i = 0; i < length; i++)
                bytes[i] = (byte)readByte(f);
            
            char[] chars = new char[readEncodings[strType].GetCharCount(bytes, 0, bytes.Length)];
            readEncodings[strType].GetChars(bytes, 0, bytes.Length, chars, 0);
            return new string(chars);
        }
        
        // Special version of readString for move routes that subtracts from lengthTemp, and in move commands, uses a different style of multibyte.
        public static string readStringMove(FileStream f, ref int lengthTemp, int strType, string source)
        {
            int length = readMultibyte(f);
            lengthTemp--;
            
            byte[] bytes = new byte[length];
            for (int i = 0; i < length; i++)
            {
                bytes[i] = (byte)readByte(f);
                lengthTemp--;
                if (bytes[i] >= 0x80)
                {
                    if (source == "Custom")
                    {
                        i++;
                        bytes[i] = (byte)readByte(f);
                        lengthTemp--;
                    }
                    else // Yep, it's different for non-custom routes. Whyyyyy
                        bytes[i] += (byte)(readByte(f) - 1);
                }
            }
            
            char[] chars = new char[readEncodings[strType].GetCharCount(bytes, 0, bytes.Length)];
            readEncodings[strType].GetChars(bytes, 0, bytes.Length, chars, 0);
            return new string(chars);
        }
        
        // Reads a database entry name string, and puts it into the given data names array.
        public static string readStringDataName(FileStream f, int id, ref string[] dataNames, int strType)
        {
            string dataName = readString(f, strType);
            if (id >= 1 && id < dataNames.Length)
                dataNames[id - 1] = dataName;
            return dataName;
        }
        
        // Reads a byte length, a count, then a list of objects.
        public static List<T> readList<T>(FileStream f, string mode = "", int arg = 0) where T : RPGByteData, new()
        {
            readMultibyte(f);
            int count = readMultibyte(f);
            List<T> list = new List<T>();
            for (int i = 0; i < count; i++)
            {
                if (mode.Equals("Page"))
                {
                    currentPage = "Page " + (i + 1);
                    currentPageNum = i + 1;
                }
                else if (mode.Equals("Pose"))
                    battlerPoseNames[arg - 1] = new string[count];
                else if (mode.Equals("Weapon"))
                    weaponAnimationNames[arg - 1] = new string[count];
                
                T obj = new T();
                obj.load(f);
                list.Add(obj);
            }
            return list;
        }
        
        // Reads a count, then a list of one-byte objects.
        public static List<T> readListNoLength<T>(FileStream f) where T : RPGData, new()
        {
            int count = readMultibyte(f);
            List<T> list = new List<T>();
            for (int i = 0; i < count; i++)
            {
                T obj = new T();
                obj.load(f);
                list.Add(obj);
            }
            return list;
        }
        
        // Reads a byte length, then a list of commands, stopping at the end command.
        public static List<Command> readCommandList(FileStream f)
        {
            readMultibyte(f); // Length
            List<Command> list = new List<Command>();
            int line = 1;
            while (true)
            {
                currentLine = "Line " + line++;
                
                Command command = new Command(f);
                list.Add(command);
                
                if (command.isEndCommand())
                    break;
            }
            return list;
        }
        
        // Reads a list of database entries.
        public static List<T> readDatabaseList<T>(FileStream f, string plural, string singular, ref string[] dataNames) where T : RPGDatabaseEntry, new()
        {
            currentEvent = plural;
            currentPage = "";
            currentEventNum = 0;
            currentPageNum = 0;
            
            int tabCount = readMultibyte(f);
            dataNames = new string[tabCount];
            
            if (plural.Equals("Battler Animations"))
            {
                battlerPoseNames = new string[tabCount][];
                weaponAnimationNames = new string[tabCount][];
            }
            
            List<T> entries = new List<T>();
            for (int i = 0; i < tabCount; i++)
            {
                if (plural.Equals("Common Events") || plural.Equals("Troops"))
                {
                    currentEvent = singular + " " + (i + 1);
                    currentEventNum = i + 1;
                }
                else
                {
                    currentLine = singular + " " + (i + 1);
                    if (plural.Equals("Battler Animations"))
                        currentBattlerAnimation = i;
                }
                
                T entry = new T();
                entry.load(f);
                entries.Add(entry);
            }
            
            return entries;
        }
        
        // Reads a byte that should be guaranteed to have the given value. If it doesn't, something went wrong, so an exception is thrown.
        public static void byteCheck(FileStream f, byte b)
        {
            byte bRead = readByte(f);
            if (b != bRead)
            {
                Console.WriteLine("Warning! A byte check failed.\n"
                                + "(At position " + hexParen(f.Position - 1) + ", read " + hexParen(bRead, 2) + ", should be " + hexParen(b, 2) + ".)\n"
                                + "Possible corrupt file or program bug.");
                throw new Exception();
            }
        }
        
        // Peeks at the next byte in the file, keeping the same position.
        public static byte bytePeek(FileStream f)
        {
            byte b = readByte(f);
            f.Seek(-1, SeekOrigin.Current);
            return b;
        }
        
        // Reads a string that should be guaranteed to have the given value using byteChecks.
        public static void stringCheck(FileStream f, string str)
        {
            byteCheck(f, (byte)stringByteLength(str));
            foreach (char c in str)
                byteCheck(f, (byte)c);
        }
        
        // Skips through a chunk if the given opcode is found. Returns an array of the bytes skipped.
        public static byte[] skipChunk(FileStream f, byte b)
        {
            long startPosition = f.Position;
            
            byte readB = readByte(f);
            if (readB >= 128)
            {
                f.Seek(-1, SeekOrigin.Current);
                readB = (byte)readMultibyte(f);
            }
            
            if (b == readB)
            {
                List<byte> bytesRead = new List<byte>();
                bytesRead.Add(b);
                
                int length = readMultibyte(f);
                bytesRead.AddRange(getMultibytesForValue(length));
                
                bytesRead.AddRange(skipBytes(f, length));
                
                return bytesRead.ToArray();
            }
            else
            {
                f.Seek(-countMultibyte(readB), SeekOrigin.Current);
                return new byte[0];
            }
        }
        
        // Runs skipChunk on a range of bytes in ascending order. Returns an array of the bytes skipped.
        public static byte[] skipChunkRange(FileStream f, byte start, byte end)
        {
            List<byte> bytesRead = new List<byte>();
            for (byte i = start; i <= end; i++)
                bytesRead.AddRange(skipChunk(f, i));
            return bytesRead.ToArray();
        }
        
        // Reads through the specified number of bytes. Returns an array of the bytes skipped.
        public static byte[] skipBytes(FileStream f, int n)
        {
            byte[] bytesRead = new byte[n];
            for (int i = 0; i < n; i++)
                bytesRead[i] = readByte(f);
            return bytesRead;
        }
        
        // Reads a byte length, then skips through that many bytes. Returns an array of the bytes skipped.
        public static byte[] skipLengthBytes(FileStream f)
        {
            int length = readMultibyte(f);
            return skipBytes(f, length);
        }
        
        /* BYTE-WRITING */
        
        // Writes a single byte to the target.
        public static void writeByte(int b)
        {
            targetWriter.Write((byte)b);
        }
        
        // Writes a value to the target as a multibyte.
        public static void writeMultibyte(int n)
        {
            writeByteArrayNoLength(getMultibytesForValue(n));
        }
        
        // Returns the bytes that should be written for this value.
        public static byte[] getMultibytesForValue(int value)
        {
            BinaryWriter bytes = new BinaryWriter(new MemoryStream());
            
            long n = (long)value;
            if (n < 0)
                n = (long)Math.Pow(2, 32) + n;
            
            if (n < 128)
                bytes.Write((byte)n);
            else
            {
                if (div(n, 128) + 128 <= 255) // Only one higher order byte will be needed.
                    bytes.Write((byte)(div(n, 128) + 128));
                else // Two or more will be needed.
                    getMultibytesForValue2(bytes, div(n, 128) + 128);
                bytes.Write((byte)(n % 128));
            }
            
            bytes.Close();
            return (bytes.BaseStream as MemoryStream).ToArray();
        }
        static void getMultibytesForValue2(BinaryWriter bytes, long n)
        {
            if (n < 128)
                bytes.Write((byte)(n + 128));
            else
            {
                if (div(n, 128) + 127 <= 255) // Only one higher order byte will be needed.
                    bytes.Write((byte)(div(n, 128) + 127));
                else // Two or more will be needed.
                    getMultibytesForValue2(bytes, div(n, 128) + 127);
                bytes.Write((byte)((n % 128) + 128));
            }
        }
        
        // Counts how many bytes would be used to write the given value as a multibyte.
        public static int countMultibyte(int n)
        {
            if (n < 128 && n >= 0)
                return 1;
            return getMultibytesForValue(n).Length;
        }
        
        // Writes a value as an int16 (two bytes).
        public static void writeTwoBytes(int value)
        {
            writeByte(value % 256);
            writeByte((value >> 8) % 256);
        }
        
        // Writes a value as an int32 (four bytes).
        public static void writeFourBytes(long value)
        {
            writeByte((int)(value % 256));
            writeByte((int)((value >> 8) % 256));
            writeByte((int)((value >> 16) % 256));
            writeByte((int)((value >> 24) % 256));
        }
        
        // Writes a byte length (always 1), then true (1) or false (0).
        public static void writeLengthBool(bool b)
        {
            writeByte(0x01);
            writeByte(b? (byte)0x01 : (byte)0x00);
        }
        
        // Writes a byte length, then a number of bytes dictated by the length.
        public static void writeLengthBytes(int value)
        {
            int length = value > 255? 2 : 1;
            writeByte(length);
            if (length == 1)
                writeByte(value);
            else
                writeTwoBytes(value);
        }
        
        // Writes a byte length, then a multibyte.
        public static void writeLengthMultibyte(int n)
        {
            writeMultibyte(countMultibyte(n));
            writeMultibyte(n);
        }
        
        // Counts how many bytes would be used to write a length (which can itself take up multiple bytes) followed by a multibyte.
        public static int countLengthMultibyte(int n)
        {
            return countMultibyte(countMultibyte(n)) + countMultibyte(n);
        }
        
        // Writes a byte length (always 8), then an eight-byte double.
        public static void writeLengthDouble(double value)
        {
            writeByte(0x08);
            targetWriter.Write(value);
        }
        
        // Writes a byte length, then a byte or two, with bits based on an array of bools.
        public static void writeLengthFlags(bool[] flags)
        {
            int total = 0;
            for (int i = 0; i < flags.Length; i++)
                if (flags[i])
                    total += (int)Math.Pow(2, i);
            
            int byteLength = flags.Length < 8? 1 : 2;
            writeByte(byteLength);
            if (byteLength == 1)
                writeByte(total);
            else
                writeTwoBytes(total);
        }
        
        // Writes a byte length, then an array of bytes.
        public static void writeByteArray(int[] bytes)
        {
            writeByte(bytes.Length);
            foreach (byte b in bytes)
                writeByte(b);
        }
        
        // Writes an array of bytes.
        public static void writeByteArrayNoLength(byte[] bytes)
        {
            foreach (byte b in bytes)
                writeByte(b);
        }
        
        // Writes a byte length, then an array of two-byte integers.
        public static void writeTwoByteArray(int[] array)
        {
            if (array.Length < 1)
            {
                writeByteArray(array);
                return;
            }
            
            writeMultibyte(array.Length * 2);
            for (int i = 0; i < array.Length; i++)
                writeTwoBytes(array[i]);
        }
        
        // Writes a byte length, then an array of four-byte integers.
        public static void writeFourByteArray(long[] array)
        {
            writeMultibyte(array.Length * 4);
            for (int i = 0; i < array.Length; i++)
                writeFourBytes(array[i]);
        }
        
        // Writes a count, then an array of multibytes.
        public static void writeMultibyteArray(int[] array)
        {
            writeMultibyte(array.Length);
            for (int i = 0; i < array.Length; i++)
                writeMultibyte(array[i]);
        }
        
        // Writes a byte length, then an 2D array of two-byte integers.
        public static void writeTwoByteArray2D(int[][] array)
        {
            writeMultibyte(array.Length * array[0].Length * 2);
            for (int i = 0; i < array.Length; i++)
                for (int j = 0; j < array[0].Length; j++)
                    writeTwoBytes(array[i][j]);
        }
        
        // Writes a byte length, then an array of bools (0s or 1s).
        public static void writeBoolArray(bool[] array)
        {
            writeMultibyte(array.Length);
            for (int i = 0; i < array.Length; i++)
                writeByte(array[i]? 1 : 0);
        }
        
        // Writes the byte length of the string followed by the string to the target. Returns lengthMinus for special multibyte reads in move commands.
        public static int writeString(string str, int strType, string source = "")
        {
            byte[] bytes = writeEncodings[strType].GetBytes(str);
            writeMultibyte(bytes.Length);
            
            int lengthMinus = 0;
            
            for (int i = 0; i < bytes.Length; i++)
            {
                if (source == "Move" && bytes[i] >= 0x80) // Non-custom move event.
                {
                    writeByte(0x81);
                    writeByte(bytes[i] - 0x80);
                    lengthMinus++;
                }
                else // Everything else.
                    writeByte(bytes[i]);
            }
            
            return lengthMinus;
        }
        
        // Writes a byte length, a count, then a list of objects.
        public static void writeList<T>(List<T> list) where T : RPGByteData, new()
        {
            int byteLength = 0;
            byteLength += countMultibyte(list.Count);
            foreach (T obj in list)
                byteLength += obj.getLength();
            
            writeMultibyte(byteLength);
            writeMultibyte(list.Count);
            for (int i = 0; i < list.Count; i++)
                list[i].write();
        }
        
        // Writes a count, then a list of objects.
        public static void writeListNoLength<T>(List<T> list) where T : RPGData, new()
        {
            writeMultibyte(list.Count);
            for (int i = 0; i < list.Count; i++)
                list[i].write();
        }
        
        // Writes a byte length, then a list of Commands.
        public static void writeCommandList(List<Command> list)
        {
            int byteLength = 0;
            foreach (Command obj in list)
                byteLength += obj.getLength();
            
            writeMultibyte(byteLength);
            foreach (Command command in list)
                command.write();
        }
        
        /* GENERAL HELPER FUNCTIONS */
        
        // Creates a text file to which the text string is written.
        public static void writeToNewFile(string filename, string text)
        {
            string dir = Path.GetDirectoryName(filename);
            if (dir != "" && !Directory.Exists(dir))
                Directory.CreateDirectory(dir);
            
            using (StreamWriter fWrite = new StreamWriter(filename, false, UNICODE))
            {
                fWrite.Write(text);
                fWrite.Close();
            }
        }
        
        // Checks if the file is in use.
        public static bool fileInUse(string filepath)
        {
            FileStream stream = null;
            
            try
            {
                if (File.Exists(filepath))
                {
                    FileInfo f = new FileInfo(filepath);
                    stream = f.Open(FileMode.Open, FileAccess.Read, FileShare.None);
                }
            }
            catch (IOException)
            {
                return true;
            }
            finally
            {
                if (stream != null)
                    stream.Close();
            }
            
            return false;
        }
        
        // Copies a file from source folder to destination folder if it exists.
        static void copyBetweenFolders(string sourcePath, string destPath, string filename)
        {
            if (File.Exists(sourcePath + "\\" + filename))
            {
                if (!File.Exists(destPath + "\\" + filename))
                    File.Copy(sourcePath + "\\" + filename, destPath + "\\" + filename);
            }
        }
        
        // Copies a directory and its contents.
        static void copyDirectory(string sourceFolder, string destFolder)
        {
            Directory.CreateDirectory(destFolder);
            IEnumerable<string> fileList = Directory.EnumerateFiles(sourceFolder, "*.*", SearchOption.TopDirectoryOnly);
            foreach (string file in fileList)
            {
                string filename = Path.GetFileName(file);
                copyBetweenFolders(sourceFolder, destFolder, filename);
            }
        }
        
        // Shuffles the values in a dictionary.
        static void shuffleDictionary<T1, T2>(ref Dictionary<T1, T2> dictionary)
        {
            List<T1> keys = new List<T1>(dictionary.Keys);
            List<T2> values = new List<T2>(dictionary.Values);
            
            int n = values.Count;
            while (n > 1)
            {
                n--;
                int k = random.Next(n + 1);
                T2 value = values[k];
                values[k] = values[n];
                values[n] = value;
            }
            
            for (int i = 0; i < keys.Count; i++)
                dictionary[keys[i]] = values[i];
        }
        
        // Shuffles a list.
        static void shuffleList<T>(ref List<T> list)
        {
            int n = list.Count;
            while (n > 1)
            {
                n--;
                int k = random.Next(n + 1);
                T value = list[k];
                list[k] = list[n];
                list[n] = value;
            }
        }
        
        // Shows "press Enter to continue" and waits for input.
        static void enterToContinue()
        {
            Console.WriteLine("Press Enter to continue.");
            Console.ReadLine();
        }
        
        // Prompts user to input a number, and limits it to given range.
        public static void numberPrompt(ref int value, int min, int max, bool blankCancels = true)
        {
            string input;
            bool valid = false;
            
            do
            {
                Console.WriteLine("Enter a number from " + min + " to " + max + ".");
                input = Console.ReadLine();
                if (input.Equals(""))
                    return;
                if (int.TryParse(input, out value))
                {
                    if (value < min)
                        value = min;
                    if (value > max)
                        value = max;
                    valid = true;
                }
            } while (!valid);
        }
        
        // Prompts user to input Y or N and returns answer.
        public static bool yesNoPrompt(bool anyNo = false)
        {
            string input;
            do
            {
                input = Console.ReadLine();
                if (input.ToUpper() == "Y")
                    return true;
                else if (anyNo || input.ToUpper() == "N")
                    return false;
            } while (input.ToUpper() != "Y" && input.ToUpper() != "N");
            
            return false;
        }
        
        // Converts byte array for multibyte to value.
        public static int convertFromMultibyte(byte[] bytes, int sum = 0)
        {
            byte b = bytes[0];
            if (b < 128)
                return b + sum;
            else
            {
                byte[] nextBytes = new byte[bytes.Length - 1];
                for (int i = 1; i < bytes.Length; i++)
                    nextBytes[i - 1] = bytes[i];
                return convertFromMultibyte(nextBytes, (b - 128 + sum) * 128);
            }
        }
        
        // Returns the integer division of two numbers.
        public static int div(int n1, int n2)
        {
            return (int)Math.Floor((double)n1 / n2);
        }
        public static long div(long n1, long n2)
        {
            return (long)Math.Floor((double)n1 / n2);
        }
        
        // Returns the value as a hex string.
        public static string hex(int value, int digits = 1)
        {
            string length = "";
            if (digits != 1)
                length = digits.ToString();
            return string.Format("{0:X" + length + "}", value);
        }
        public static string hex(long value, int digits = 1)
        {
            return hex((int)value);
        }
        
        // Returns the value with the hex string after it in parentheses (0x00).
        public static string hexParen(int value, int digits = 1)
        {
            return value + " (0x" + hex(value, digits) + ")";
        }
        public static string hexParen(long value, int digits = 1)
        {
            return hexParen((int)value, digits);
        }
        
        // Translates bool to "on" or "off" string.
        public static string onOff(bool value)
        {
            return value? "ON" : "OFF";
        }
        
        // Centers string for 50-character-wide box.
        public static string centeredText(string str)
        {
            int originalLength = str.Length;
            int spaces = 0;
            while (originalLength + (spaces * 2) < 50)
            {
                str = " " + str;
                spaces++;
            }
            return str;
        }
        
        // Returns whether the string can be reasonably interpreted as true or false.
        public static bool validTrueFalse(string value)
        {
            value = value.ToLower().Trim();
            if (value.Equals("y") || value.Equals("n")
             || value.Equals("yes") || value.Equals("no")
             || value.Equals("1") || value.Equals("0")
             || value.Equals("true") || value.Equals("false"))
                return true;
            return false;
        }
        
        // Parses a string as a bool.
        public static bool boolString(string value)
        {
            value = value.ToLower().Trim();
            if (value.Equals("y") || value.Equals("yes") || value.Equals("1") || value.Equals("true"))
                return true;
            return false;
        }
        
        // Returns whether a string would be a valid filename.
        public static bool isValidFilename(string str)
        {
            char[] invalidChars = Path.GetInvalidFileNameChars();
            foreach(char c in invalidChars)
            {
                if (str.IndexOf(c) != -1)
                    return false;
            }
            return true;
        }
        
        // Gets byte length of string (using "to translate" write encoding), relevant for character limits.
        public static int stringByteLength(string str)
        {
            return writeEncodings[S_TOTRANSLATE].GetBytes(str).Length;
        }
        
        // Returns a string describing the current position of file processing.
        public static string currentPosition(bool includeLine = true)
        {
            return currentFile + " " + currentEvent + (currentPage != ""? " " + currentPage : "") + (includeLine? " " + currentLine : "");
        }
        
        // Shows a message if debug mode is enabled.
        public static void debugMessage(string str)
        {
            if (debugMode)
                Console.WriteLine(str);
        }
    }
}
